#!/usr/bin/env python3
"""
MERSAD Windows Agent - GUI Edition
Cyber Attack Simulation Platform

This agent runs on the target Windows machine with a graphical interface.
Double-click to run - no command line needed!

Created by: Engineer Malek Shashaa
Email: malekshashaa1993@gmail.com
"""

import os
import sys
import json
import time
import socket
import platform
import subprocess
import threading
import sqlite3
import webbrowser
from datetime import datetime
from http.server import HTTPServer, BaseHTTPRequestHandler
from urllib.request import urlopen, Request
from urllib.error import URLError
import tkinter as tk
from tkinter import ttk, messagebox, scrolledtext

AGENT_VERSION = "1.2.0"
AGENT_PORT = 8888
DB_FILE = "mersad_agent.db"

class Database:
    """SQLite database for attack history"""
    
    def __init__(self, db_path=None):
        if db_path is None:
            script_dir = os.path.dirname(os.path.abspath(__file__))
            db_path = os.path.join(script_dir, DB_FILE)
        self.db_path = db_path
        self.init_db()
    
    def init_db(self):
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute('''CREATE TABLE IF NOT EXISTS attacks (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            attack_id TEXT,
            name TEXT,
            category TEXT,
            mitre_id TEXT,
            timestamp TEXT,
            executed INTEGER,
            success INTEGER,
            detected INTEGER,
            output TEXT,
            errors TEXT,
            synced INTEGER DEFAULT 0
        )''')
        c.execute('''CREATE TABLE IF NOT EXISTS settings (
            key TEXT PRIMARY KEY,
            value TEXT
        )''')
        c.execute('''CREATE TABLE IF NOT EXISTS sync_log (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            timestamp TEXT,
            direction TEXT,
            status TEXT,
            message TEXT
        )''')
        conn.commit()
        conn.close()
    
    def add_attack(self, attack_data):
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        
        executed = attack_data.get('executed', False)
        success = attack_data.get('success', False)
        detected = attack_data.get('detected')
        
        c.execute('''INSERT INTO attacks 
            (attack_id, name, category, mitre_id, timestamp, executed, success, detected, output, errors, synced)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)''',
            (attack_data.get('attack_id', ''),
             attack_data.get('name', ''),
             attack_data.get('category', ''),
             attack_data.get('mitre_id', ''),
             attack_data.get('timestamp', datetime.utcnow().isoformat()),
             1 if executed else 0,
             1 if success else 0,
             1 if detected else (0 if detected is False else None),
             json.dumps(attack_data.get('outputs', [])),
             json.dumps(attack_data.get('errors', [])),
             0))
        attack_db_id = c.lastrowid
        conn.commit()
        conn.close()
        return attack_db_id
    
    def get_attacks(self, limit=100):
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute('SELECT * FROM attacks ORDER BY id DESC LIMIT ?', (limit,))
        rows = c.fetchall()
        conn.close()
        
        attacks = []
        for row in rows:
            attacks.append({
                'db_id': row[0],
                'attack_id': row[1],
                'name': row[2],
                'category': row[3],
                'mitre_id': row[4],
                'timestamp': row[5],
                'executed': bool(row[6]),
                'success': bool(row[7]),
                'detected': bool(row[8]) if row[8] is not None else None,
                'outputs': json.loads(row[9]) if row[9] else [],
                'errors': json.loads(row[10]) if row[10] else [],
                'synced': bool(row[11])
            })
        return attacks
    
    def get_stats(self):
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute('SELECT COUNT(*) FROM attacks')
        total = c.fetchone()[0]
        c.execute('SELECT COUNT(*) FROM attacks WHERE success = 1')
        successful = c.fetchone()[0]
        c.execute('SELECT COUNT(*) FROM attacks WHERE success = 0')
        failed = c.fetchone()[0]
        c.execute('SELECT COUNT(*) FROM attacks WHERE detected = 1')
        detected = c.fetchone()[0]
        conn.close()
        return {'total': total, 'successful': successful, 'failed': failed, 'detected': detected}
    
    def mark_synced(self, attack_ids):
        if not attack_ids:
            return
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        placeholders = ','.join('?' * len(attack_ids))
        c.execute(f'UPDATE attacks SET synced = 1 WHERE id IN ({placeholders})', attack_ids)
        conn.commit()
        conn.close()
    
    def get_unsynced(self):
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute('SELECT * FROM attacks WHERE synced = 0 ORDER BY id ASC')
        rows = c.fetchall()
        conn.close()
        return rows
    
    def get_setting(self, key, default=None):
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute('SELECT value FROM settings WHERE key = ?', (key,))
        row = c.fetchone()
        conn.close()
        return row[0] if row else default
    
    def set_setting(self, key, value):
        conn = sqlite3.connect(self.db_path)
        c = conn.cursor()
        c.execute('INSERT OR REPLACE INTO settings (key, value) VALUES (?, ?)', (key, value))
        conn.commit()
        conn.close()


class AttackSimulator:
    """Executes attack simulations on Windows"""
    
    def __init__(self, db):
        self.db = db
        self.results = []
        self.is_windows = platform.system() == "Windows"
        
    def execute_attack(self, attack_id, attack_config):
        """Execute a specific attack simulation"""
        result = {
            "attack_id": attack_id,
            "name": attack_config.get("name", "Unknown"),
            "category": attack_config.get("category", "Unknown"),
            "mitre_id": attack_config.get("mitre_id", "Unknown"),
            "timestamp": datetime.utcnow().isoformat() + "Z",
            "executed": False,
            "success": False,
            "detected": None,
            "outputs": [],
            "errors": []
        }
        
        commands = attack_config.get("commands", [])
        
        all_succeeded = True
        any_executed = False
        
        for cmd in commands:
            try:
                proc = subprocess.run(
                    cmd,
                    shell=True,
                    capture_output=True,
                    text=True,
                    timeout=30
                )
                
                cmd_success = proc.returncode == 0
                result["outputs"].append({
                    "command": cmd,
                    "stdout": proc.stdout[:1000] if proc.stdout else "",
                    "stderr": proc.stderr[:500] if proc.stderr else "",
                    "returncode": proc.returncode,
                    "success": cmd_success
                })
                any_executed = True
                if not cmd_success:
                    all_succeeded = False
                
            except subprocess.TimeoutExpired:
                result["errors"].append({"command": cmd, "error": "Timeout"})
                all_succeeded = False
            except Exception as e:
                result["errors"].append({"command": cmd, "error": str(e)})
                all_succeeded = False
        
        result["executed"] = any_executed
        result["success"] = any_executed and all_succeeded
        
        self.results.append(result)
        self.db.add_attack(result)
        return result
    
    def get_system_info(self):
        """Get information about the target system"""
        info = {
            "hostname": socket.gethostname(),
            "platform": platform.system(),
            "platform_version": platform.version(),
            "architecture": platform.machine(),
            "processor": platform.processor(),
            "python_version": sys.version,
            "agent_version": AGENT_VERSION
        }
        
        try:
            import psutil
            info["cpu_count"] = psutil.cpu_count()
            info["memory_total_gb"] = round(psutil.virtual_memory().total / (1024**3), 2)
            info["disk_total_gb"] = round(psutil.disk_usage('/').total / (1024**3), 2)
        except ImportError:
            pass
        
        return info


class AgentHandler(BaseHTTPRequestHandler):
    """HTTP handler for agent commands"""
    
    simulator = None
    db = None
    controller_url = None
    gui_callback = None
    
    def log_message(self, format, *args):
        msg = f"[{datetime.now().strftime('%H:%M:%S')}] {args[0]}"
        print(msg)
        if self.gui_callback:
            self.gui_callback('log', msg)
    
    def send_json_response(self, data, status=200):
        self.send_response(status)
        self.send_header('Content-Type', 'application/json')
        self.send_header('Access-Control-Allow-Origin', '*')
        self.end_headers()
        self.wfile.write(json.dumps(data).encode())
    
    def do_OPTIONS(self):
        self.send_response(200)
        self.send_header('Access-Control-Allow-Origin', '*')
        self.send_header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
        self.send_header('Access-Control-Allow-Headers', 'Content-Type')
        self.end_headers()
    
    def do_GET(self):
        if self.path == '/status':
            stats = self.db.get_stats() if self.db else {}
            self.send_json_response({
                "status": "online",
                "system_info": self.simulator.get_system_info() if self.simulator else {},
                "stats": stats,
                "results_count": len(self.simulator.results) if self.simulator else 0
            })
        elif self.path == '/results':
            attacks = self.db.get_attacks(100) if self.db else []
            self.send_json_response({"results": attacks})
        elif self.path == '/stats':
            stats = self.db.get_stats() if self.db else {}
            self.send_json_response(stats)
        else:
            self.send_json_response({"error": "Not found"}, 404)
    
    def do_POST(self):
        content_length = int(self.headers.get('Content-Length', 0))
        body = self.rfile.read(content_length).decode()
        
        try:
            data = json.loads(body) if body else {}
        except json.JSONDecodeError:
            self.send_json_response({"error": "Invalid JSON"}, 400)
            return
        
        if self.path == '/execute':
            attack_id = data.get('attack_id')
            attack_config = data.get('attack_config', {})
            
            if not attack_config and 'commands' in data:
                attack_config = {
                    'name': data.get('name', attack_id or 'Unknown'),
                    'category': data.get('category', 'Unknown'),
                    'mitre_id': data.get('mitre_id', 'Unknown'),
                    'commands': data.get('commands', [])
                }
            
            if not attack_id:
                self.send_json_response({"error": "attack_id required"}, 400)
                return
            
            result = self.simulator.execute_attack(attack_id, attack_config)
            
            if self.gui_callback:
                self.gui_callback('attack', result)
            
            if self.controller_url:
                self.report_to_controller(result)
            
            self.send_json_response({"success": True, "result": result})
            
        elif self.path == '/execute-batch':
            attacks = data.get('attacks', [])
            results = []
            
            for attack in attacks:
                result = self.simulator.execute_attack(
                    attack.get('attack_id'),
                    attack.get('attack_config', {})
                )
                results.append(result)
                if self.gui_callback:
                    self.gui_callback('attack', result)
            
            if self.controller_url:
                self.report_batch_to_controller(results)
            
            self.send_json_response({"success": True, "results": results})
            
        elif self.path == '/clear':
            self.simulator.results = []
            self.send_json_response({"success": True, "message": "Results cleared"})
            
        elif self.path == '/mark-detected':
            attack_id = data.get('attack_id')
            detected = data.get('detected', True)
            
            for result in self.simulator.results:
                if result.get('attack_id') == attack_id:
                    result['detected'] = detected
                    break
            
            self.send_json_response({"success": True})
        else:
            self.send_json_response({"error": "Not found"}, 404)
    
    def report_to_controller(self, result):
        """Report attack result back to controller"""
        try:
            req = Request(
                f"{self.controller_url}/api/agent/report",
                data=json.dumps(result).encode(),
                headers={'Content-Type': 'application/json'},
                method='POST'
            )
            urlopen(req, timeout=5)
        except Exception as e:
            print(f"Failed to report to controller: {e}")
    
    def report_batch_to_controller(self, results):
        """Report batch results to controller"""
        try:
            req = Request(
                f"{self.controller_url}/api/agent/report-batch",
                data=json.dumps({"results": results}).encode(),
                headers={'Content-Type': 'application/json'},
                method='POST'
            )
            urlopen(req, timeout=10)
        except Exception as e:
            print(f"Failed to report batch to controller: {e}")


class SyncManager:
    """Handles synchronization with Kali attacker"""
    
    def __init__(self, db, kali_url=None):
        self.db = db
        self.kali_url = kali_url
        self.sync_interval = 10
        self.running = False
        self.thread = None
    
    def set_kali_url(self, url):
        self.kali_url = url
        self.db.set_setting('kali_url', url)
    
    def start(self):
        if not self.running:
            self.running = True
            self.thread = threading.Thread(target=self._sync_loop, daemon=True)
            self.thread.start()
    
    def stop(self):
        self.running = False
    
    def _sync_loop(self):
        while self.running:
            if self.kali_url:
                self.sync_results()
            time.sleep(self.sync_interval)
    
    def sync_results(self):
        """Push unsynced attack results to Kali attacker"""
        if not self.kali_url:
            return False
        
        unsynced = self.db.get_unsynced()
        if not unsynced:
            return True
        
        results = []
        ids = []
        for row in unsynced:
            results.append({
                'attack_id': row[1],
                'name': row[2],
                'category': row[3],
                'mitre_id': row[4],
                'timestamp': row[5],
                'executed': bool(row[6]),
                'success': bool(row[7]),
                'detected': bool(row[8]) if row[8] is not None else None,
                'outputs': json.loads(row[9]) if row[9] else [],
                'errors': json.loads(row[10]) if row[10] else []
            })
            ids.append(row[0])
        
        try:
            req = Request(
                f"{self.kali_url}/api/sync/receive",
                data=json.dumps({'results': results, 'agent_id': socket.gethostname()}).encode(),
                headers={'Content-Type': 'application/json'},
                method='POST'
            )
            response = urlopen(req, timeout=10)
            data = json.loads(response.read().decode())
            if data.get('success'):
                self.db.mark_synced(ids)
                return True
        except Exception as e:
            print(f"Sync failed: {e}")
        
        return False


class MersadAgentGUI:
    """Main GUI application for Windows Agent"""
    
    def __init__(self):
        self.root = tk.Tk()
        self.root.title("MERSAD Agent - مرصاد")
        self.root.geometry("900x700")
        self.root.configure(bg='#1a1a2e')
        
        self.db = Database()
        self.simulator = AttackSimulator(self.db)
        self.sync_manager = SyncManager(self.db)
        
        self.server = None
        self.server_thread = None
        
        self.setup_styles()
        self.create_widgets()
        self.load_settings()
        self.start_server()
        self.update_stats()
        
        self.refresh_timer()
    
    def setup_styles(self):
        style = ttk.Style()
        style.theme_use('clam')
        
        style.configure('TFrame', background='#1a1a2e')
        style.configure('TLabel', background='#1a1a2e', foreground='#e0e0e0', font=('Segoe UI', 10))
        style.configure('Title.TLabel', font=('Segoe UI', 24, 'bold'), foreground='#00d4ff')
        style.configure('Stat.TLabel', font=('Segoe UI', 18, 'bold'), foreground='#00ff88')
        style.configure('TButton', font=('Segoe UI', 10), padding=10)
        style.configure('Success.TLabel', foreground='#00ff88')
        style.configure('Error.TLabel', foreground='#ff4444')
        style.configure('Warning.TLabel', foreground='#ffaa00')
        
        style.configure('Treeview', 
                        background='#16213e',
                        foreground='#e0e0e0',
                        fieldbackground='#16213e',
                        font=('Segoe UI', 9))
        style.configure('Treeview.Heading', 
                        background='#0f3460',
                        foreground='#00d4ff',
                        font=('Segoe UI', 10, 'bold'))
        style.map('Treeview', background=[('selected', '#0f3460')])
    
    def create_widgets(self):
        main_frame = ttk.Frame(self.root)
        main_frame.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
        
        header = ttk.Frame(main_frame)
        header.pack(fill=tk.X, pady=(0, 20))
        
        title_label = ttk.Label(header, text="مرصاد MERSAD Agent", style='Title.TLabel')
        title_label.pack(side=tk.LEFT)
        
        version_label = ttk.Label(header, text=f"v{AGENT_VERSION}", style='TLabel')
        version_label.pack(side=tk.LEFT, padx=10)
        
        self.status_label = ttk.Label(header, text="● Online", style='Success.TLabel')
        self.status_label.pack(side=tk.RIGHT)
        
        stats_frame = ttk.Frame(main_frame)
        stats_frame.pack(fill=tk.X, pady=(0, 20))
        
        self.stat_widgets = {}
        stat_configs = [
            ('total', 'Total Attacks', '#00d4ff'),
            ('successful', 'Successful', '#00ff88'),
            ('failed', 'Failed', '#ff4444'),
            ('detected', 'Detected', '#ffaa00')
        ]
        
        for key, label, color in stat_configs:
            frame = ttk.Frame(stats_frame)
            frame.pack(side=tk.LEFT, expand=True, padx=10)
            
            value_label = tk.Label(frame, text="0", font=('Segoe UI', 28, 'bold'), 
                                   fg=color, bg='#1a1a2e')
            value_label.pack()
            
            name_label = ttk.Label(frame, text=label)
            name_label.pack()
            
            self.stat_widgets[key] = value_label
        
        notebook = ttk.Notebook(main_frame)
        notebook.pack(fill=tk.BOTH, expand=True)
        
        history_frame = ttk.Frame(notebook)
        notebook.add(history_frame, text="Attack History")
        
        columns = ('Time', 'Name', 'MITRE ID', 'Category', 'Status', 'Synced')
        self.history_tree = ttk.Treeview(history_frame, columns=columns, show='headings', height=15)
        
        for col in columns:
            self.history_tree.heading(col, text=col)
            width = 150 if col == 'Name' else 100
            self.history_tree.column(col, width=width)
        
        scrollbar = ttk.Scrollbar(history_frame, orient=tk.VERTICAL, command=self.history_tree.yview)
        self.history_tree.configure(yscrollcommand=scrollbar.set)
        
        self.history_tree.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        
        self.history_tree.bind('<Double-1>', self.show_attack_details)
        
        settings_frame = ttk.Frame(notebook)
        notebook.add(settings_frame, text="Settings")
        
        settings_inner = ttk.Frame(settings_frame)
        settings_inner.pack(pady=20, padx=20, fill=tk.X)
        
        ttk.Label(settings_inner, text="Agent Port:").grid(row=0, column=0, sticky=tk.W, pady=5)
        self.port_var = tk.StringVar(value=str(AGENT_PORT))
        port_entry = ttk.Entry(settings_inner, textvariable=self.port_var, width=10)
        port_entry.grid(row=0, column=1, sticky=tk.W, pady=5, padx=10)
        
        ttk.Label(settings_inner, text="Kali Controller URL:").grid(row=1, column=0, sticky=tk.W, pady=5)
        self.kali_url_var = tk.StringVar()
        kali_entry = ttk.Entry(settings_inner, textvariable=self.kali_url_var, width=40)
        kali_entry.grid(row=1, column=1, sticky=tk.W, pady=5, padx=10)
        
        ttk.Label(settings_inner, text="(e.g., http://192.168.1.100:8080)").grid(row=1, column=2, sticky=tk.W)
        
        btn_frame = ttk.Frame(settings_inner)
        btn_frame.grid(row=2, column=0, columnspan=3, pady=20)
        
        save_btn = ttk.Button(btn_frame, text="Save Settings", command=self.save_settings)
        save_btn.pack(side=tk.LEFT, padx=5)
        
        sync_btn = ttk.Button(btn_frame, text="Sync Now", command=self.manual_sync)
        sync_btn.pack(side=tk.LEFT, padx=5)
        
        test_btn = ttk.Button(btn_frame, text="Test Connection", command=self.test_kali_connection)
        test_btn.pack(side=tk.LEFT, padx=5)
        
        log_frame = ttk.Frame(notebook)
        notebook.add(log_frame, text="Logs")
        
        self.log_text = scrolledtext.ScrolledText(log_frame, height=20, bg='#0a0a15', fg='#00ff88',
                                                   font=('Consolas', 10))
        self.log_text.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)
        
        about_frame = ttk.Frame(notebook)
        notebook.add(about_frame, text="About")
        
        about_inner = ttk.Frame(about_frame)
        about_inner.pack(pady=40)
        
        ttk.Label(about_inner, text="MERSAD مرصاد", style='Title.TLabel').pack(pady=10)
        ttk.Label(about_inner, text="Cyber Attack Simulation Platform").pack()
        ttk.Label(about_inner, text=f"Agent Version: {AGENT_VERSION}").pack(pady=10)
        ttk.Label(about_inner, text="").pack()
        ttk.Label(about_inner, text="Created by: Engineer Malek Shashaa").pack()
        ttk.Label(about_inner, text="Email: malekshashaa1993@gmail.com").pack()
        ttk.Label(about_inner, text="").pack(pady=10)
        ttk.Label(about_inner, text="For authorized security testing only").pack()
        
        footer = ttk.Frame(main_frame)
        footer.pack(fill=tk.X, pady=(20, 0))
        
        self.ip_label = ttk.Label(footer, text=f"Listening: {socket.gethostbyname(socket.gethostname())}:{AGENT_PORT}")
        self.ip_label.pack(side=tk.LEFT)
        
        self.sync_status = ttk.Label(footer, text="Sync: Not configured", style='Warning.TLabel')
        self.sync_status.pack(side=tk.RIGHT)
    
    def gui_callback(self, event_type, data):
        """Handle events from HTTP server"""
        if event_type == 'attack':
            self.root.after(0, self.update_stats)
            self.root.after(0, self.refresh_history)
        elif event_type == 'log':
            self.root.after(0, lambda: self.add_log(data))
    
    def add_log(self, message):
        self.log_text.insert(tk.END, message + "\n")
        self.log_text.see(tk.END)
    
    def start_server(self):
        """Start the HTTP server in a background thread"""
        port = int(self.port_var.get())
        
        AgentHandler.simulator = self.simulator
        AgentHandler.db = self.db
        AgentHandler.gui_callback = self.gui_callback
        
        def run_server():
            self.server = HTTPServer(('0.0.0.0', port), AgentHandler)
            self.add_log(f"Server started on port {port}")
            self.server.serve_forever()
        
        self.server_thread = threading.Thread(target=run_server, daemon=True)
        self.server_thread.start()
    
    def update_stats(self):
        stats = self.db.get_stats()
        for key, widget in self.stat_widgets.items():
            widget.config(text=str(stats.get(key, 0)))
    
    def refresh_history(self):
        for item in self.history_tree.get_children():
            self.history_tree.delete(item)
        
        attacks = self.db.get_attacks(100)
        for attack in attacks:
            status = "Success" if attack['success'] else "Failed"
            synced = "Yes" if attack['synced'] else "No"
            timestamp = attack['timestamp'][:19] if attack['timestamp'] else ""
            
            tags = ('success',) if attack['success'] else ('failed',)
            self.history_tree.insert('', 0, values=(
                timestamp,
                attack['name'],
                attack['mitre_id'],
                attack['category'],
                status,
                synced
            ), tags=tags)
        
        self.history_tree.tag_configure('success', foreground='#00ff88')
        self.history_tree.tag_configure('failed', foreground='#ff4444')
    
    def show_attack_details(self, event):
        selection = self.history_tree.selection()
        if not selection:
            return
        
        item = self.history_tree.item(selection[0])
        values = item['values']
        
        detail_window = tk.Toplevel(self.root)
        detail_window.title(f"Attack Details - {values[1]}")
        detail_window.geometry("600x400")
        detail_window.configure(bg='#1a1a2e')
        
        text = scrolledtext.ScrolledText(detail_window, bg='#0a0a15', fg='#e0e0e0',
                                         font=('Consolas', 10))
        text.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)
        
        attacks = self.db.get_attacks(100)
        for attack in attacks:
            if attack['name'] == values[1] and attack['timestamp'][:19] == values[0]:
                text.insert(tk.END, f"Attack: {attack['name']}\n")
                text.insert(tk.END, f"MITRE ID: {attack['mitre_id']}\n")
                text.insert(tk.END, f"Category: {attack['category']}\n")
                text.insert(tk.END, f"Status: {'Success' if attack['success'] else 'Failed'}\n")
                text.insert(tk.END, f"Timestamp: {attack['timestamp']}\n\n")
                text.insert(tk.END, "--- Outputs ---\n")
                for output in attack['outputs']:
                    text.insert(tk.END, f"Command: {output.get('command', '')}\n")
                    text.insert(tk.END, f"Return Code: {output.get('returncode', '')}\n")
                    if output.get('stdout'):
                        text.insert(tk.END, f"Stdout:\n{output['stdout']}\n")
                    if output.get('stderr'):
                        text.insert(tk.END, f"Stderr:\n{output['stderr']}\n")
                    text.insert(tk.END, "\n")
                if attack['errors']:
                    text.insert(tk.END, "--- Errors ---\n")
                    for error in attack['errors']:
                        text.insert(tk.END, f"{error}\n")
                break
    
    def load_settings(self):
        kali_url = self.db.get_setting('kali_url', '')
        self.kali_url_var.set(kali_url)
        if kali_url:
            self.sync_manager.set_kali_url(kali_url)
            AgentHandler.controller_url = kali_url
            self.sync_status.config(text=f"Sync: {kali_url}", style='Success.TLabel')
            self.sync_manager.start()
    
    def save_settings(self):
        kali_url = self.kali_url_var.get().strip()
        if kali_url:
            self.sync_manager.set_kali_url(kali_url)
            AgentHandler.controller_url = kali_url
            self.sync_status.config(text=f"Sync: {kali_url}", style='Success.TLabel')
            self.sync_manager.start()
            messagebox.showinfo("Settings", "Settings saved successfully!")
        else:
            self.sync_status.config(text="Sync: Not configured", style='Warning.TLabel')
            messagebox.showinfo("Settings", "Settings saved (no Kali URL configured)")
    
    def manual_sync(self):
        if not self.kali_url_var.get():
            messagebox.showwarning("Sync", "Please configure Kali Controller URL first")
            return
        
        if self.sync_manager.sync_results():
            messagebox.showinfo("Sync", "Sync completed successfully!")
            self.refresh_history()
        else:
            messagebox.showerror("Sync", "Sync failed. Check the connection.")
    
    def test_kali_connection(self):
        kali_url = self.kali_url_var.get().strip()
        if not kali_url:
            messagebox.showwarning("Test", "Please enter Kali Controller URL first")
            return
        
        try:
            req = Request(f"{kali_url}/api/health", method='GET')
            response = urlopen(req, timeout=5)
            data = json.loads(response.read().decode())
            messagebox.showinfo("Connection Test", f"Connected successfully!\nStatus: {data.get('status', 'OK')}")
        except Exception as e:
            messagebox.showerror("Connection Test", f"Connection failed:\n{str(e)}")
    
    def refresh_timer(self):
        self.update_stats()
        self.refresh_history()
        self.root.after(5000, self.refresh_timer)
    
    def run(self):
        self.root.mainloop()


def main():
    app = MersadAgentGUI()
    app.run()


if __name__ == '__main__':
    main()
